# Copyright 2015 Google Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

require 'google/apis/core/base_service'
require 'google/apis/core/json_representation'
require 'google/apis/core/hashable'
require 'google/apis/errors'

module Google
  module Apis
    module ComputeV1
      # Compute Engine API
      #
      # Creates and runs virtual machines on Google Cloud Platform.
      #
      # @example
      #    require 'google/apis/compute_v1'
      #
      #    Compute = Google::Apis::ComputeV1 # Alias the module
      #    service = Compute::ComputeService.new
      #
      # @see https://developers.google.com/compute/docs/reference/latest/
      class ComputeService < Google::Apis::Core::BaseService
        # @return [String]
        #  API key. Your API key identifies your project and provides you with API access,
        #  quota, and reports. Required unless you provide an OAuth 2.0 token.
        attr_accessor :key

        # @return [String]
        #  An opaque string that represents a user for quota purposes. Must not exceed 40
        #  characters.
        attr_accessor :quota_user

        # @return [String]
        #  Deprecated. Please use quotaUser instead.
        attr_accessor :user_ip

        def initialize
          super('https://compute.googleapis.com/', 'compute/v1/projects/')
          @batch_path = 'batch/compute/v1'
        end
        
        # Retrieves an aggregated list of accelerator types.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AcceleratorTypeAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AcceleratorTypeAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_accelerator_type_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/acceleratorTypes', options)
          command.response_representation = Google::Apis::ComputeV1::AcceleratorTypeAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::AcceleratorTypeAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified accelerator type.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] accelerator_type
        #   Name of the accelerator type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AcceleratorType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AcceleratorType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_accelerator_type(project, zone, accelerator_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/acceleratorTypes/{acceleratorType}', options)
          command.response_representation = Google::Apis::ComputeV1::AcceleratorType::Representation
          command.response_class = Google::Apis::ComputeV1::AcceleratorType
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['acceleratorType'] = accelerator_type unless accelerator_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of accelerator types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AcceleratorTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AcceleratorTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_accelerator_types(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/acceleratorTypes', options)
          command.response_representation = Google::Apis::ComputeV1::AcceleratorTypeList::Representation
          command.response_class = Google::Apis::ComputeV1::AcceleratorTypeList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of addresses.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AddressAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AddressAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_addresses(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/addresses', options)
          command.response_representation = Google::Apis::ComputeV1::AddressAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::AddressAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified address resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] address
        #   Name of the address resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_address(project, region, address, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/regions/{region}/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['address'] = address unless address.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified address resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] address
        #   Name of the address resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Address] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Address]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_address(project, region, address, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeV1::Address::Representation
          command.response_class = Google::Apis::ComputeV1::Address
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['address'] = address unless address.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an address resource in the specified project by using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [Google::Apis::ComputeV1::Address] address_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_address(project, region, address_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/addresses', options)
          command.request_representation = Google::Apis::ComputeV1::Address::Representation
          command.request_object = address_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of addresses contained within the specified region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AddressList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AddressList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_addresses(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/addresses', options)
          command.response_representation = Google::Apis::ComputeV1::AddressList::Representation
          command.response_class = Google::Apis::ComputeV1::AddressList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of autoscalers.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AutoscalerAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AutoscalerAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_autoscalers(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/autoscalers', options)
          command.response_representation = Google::Apis::ComputeV1::AutoscalerAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::AutoscalerAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified autoscaler.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [String] autoscaler
        #   Name of the autoscaler to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_autoscaler(project, zone, autoscaler, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/zones/{zone}/autoscalers/{autoscaler}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified autoscaler resource. Gets a list of available
        # autoscalers by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [String] autoscaler
        #   Name of the autoscaler to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Autoscaler] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Autoscaler]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_autoscaler(project, zone, autoscaler, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/autoscalers/{autoscaler}', options)
          command.response_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.response_class = Google::Apis::ComputeV1::Autoscaler
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an autoscaler in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [Google::Apis::ComputeV1::Autoscaler] autoscaler_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_autoscaler(project, zone, autoscaler_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of autoscalers contained within the specified zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AutoscalerList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AutoscalerList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_autoscalers(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/autoscalers', options)
          command.response_representation = Google::Apis::ComputeV1::AutoscalerList::Representation
          command.response_class = Google::Apis::ComputeV1::AutoscalerList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an autoscaler in the specified project using the data included in the
        # request. This method supports PATCH semantics and uses the JSON merge patch
        # format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [Google::Apis::ComputeV1::Autoscaler] autoscaler_object
        # @param [String] autoscaler
        #   Name of the autoscaler to patch.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_autoscaler(project, zone, autoscaler_object = nil, autoscaler: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/zones/{zone}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an autoscaler in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   Name of the zone for this request.
        # @param [Google::Apis::ComputeV1::Autoscaler] autoscaler_object
        # @param [String] autoscaler
        #   Name of the autoscaler to update.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_autoscaler(project, zone, autoscaler_object = nil, autoscaler: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, '{project}/zones/{zone}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds a key for validating requests with signed URLs for this backend bucket.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to which the Signed URL Key should be added.
        #   The name should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::SignedUrlKey] signed_url_key_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_backend_bucket_signed_url_key(project, backend_bucket, signed_url_key_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/backendBuckets/{backendBucket}/addSignedUrlKey', options)
          command.request_representation = Google::Apis::ComputeV1::SignedUrlKey::Representation
          command.request_object = signed_url_key_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified BackendBucket resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_bucket(project, backend_bucket, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/backendBuckets/{backendBucket}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a key for validating requests with signed URLs for this backend bucket.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to which the Signed URL Key should be added.
        #   The name should conform to RFC1035.
        # @param [String] key_name
        #   The name of the Signed URL Key to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_bucket_signed_url_key(project, backend_bucket, key_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/backendBuckets/{backendBucket}/deleteSignedUrlKey', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['keyName'] = key_name unless key_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified BackendBucket resource. Gets a list of available backend
        # buckets by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendBucket] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendBucket]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_bucket(project, backend_bucket, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/backendBuckets/{backendBucket}', options)
          command.response_representation = Google::Apis::ComputeV1::BackendBucket::Representation
          command.response_class = Google::Apis::ComputeV1::BackendBucket
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a BackendBucket resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::BackendBucket] backend_bucket_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_backend_bucket(project, backend_bucket_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/backendBuckets', options)
          command.request_representation = Google::Apis::ComputeV1::BackendBucket::Representation
          command.request_object = backend_bucket_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of BackendBucket resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendBucketList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendBucketList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_backend_buckets(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/backendBuckets', options)
          command.response_representation = Google::Apis::ComputeV1::BackendBucketList::Representation
          command.response_class = Google::Apis::ComputeV1::BackendBucketList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified BackendBucket resource with the data included in the
        # request. This method supports PATCH semantics and uses the JSON merge patch
        # format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to patch.
        # @param [Google::Apis::ComputeV1::BackendBucket] backend_bucket_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_backend_bucket(project, backend_bucket, backend_bucket_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/global/backendBuckets/{backendBucket}', options)
          command.request_representation = Google::Apis::ComputeV1::BackendBucket::Representation
          command.request_object = backend_bucket_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified BackendBucket resource with the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_bucket
        #   Name of the BackendBucket resource to update.
        # @param [Google::Apis::ComputeV1::BackendBucket] backend_bucket_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_backend_bucket(project, backend_bucket, backend_bucket_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, '{project}/global/backendBuckets/{backendBucket}', options)
          command.request_representation = Google::Apis::ComputeV1::BackendBucket::Representation
          command.request_object = backend_bucket_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendBucket'] = backend_bucket unless backend_bucket.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds a key for validating requests with signed URLs for this backend service.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to which the Signed URL Key should be
        #   added. The name should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::SignedUrlKey] signed_url_key_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_backend_service_signed_url_key(project, backend_service, signed_url_key_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/backendServices/{backendService}/addSignedUrlKey', options)
          command.request_representation = Google::Apis::ComputeV1::SignedUrlKey::Representation
          command.request_object = signed_url_key_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of all BackendService resources, regional and global,
        # available to the specified project.
        # @param [String] project
        #   Name of the project scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendServiceAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendServiceAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_backend_service_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/backendServices', options)
          command.response_representation = Google::Apis::ComputeV1::BackendServiceAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::BackendServiceAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified BackendService resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_service(project, backend_service, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/backendServices/{backendService}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes a key for validating requests with signed URLs for this backend
        # service.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to which the Signed URL Key should be
        #   added. The name should conform to RFC1035.
        # @param [String] key_name
        #   The name of the Signed URL Key to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_backend_service_signed_url_key(project, backend_service, key_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/backendServices/{backendService}/deleteSignedUrlKey', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['keyName'] = key_name unless key_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified BackendService resource. Gets a list of available
        # backend services.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendService] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendService]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_service(project, backend_service, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/backendServices/{backendService}', options)
          command.response_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.response_class = Google::Apis::ComputeV1::BackendService
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the most recent health check results for this BackendService.
        # @param [String] project
        # @param [String] backend_service
        #   Name of the BackendService resource to which the queried instance belongs.
        # @param [Google::Apis::ComputeV1::ResourceGroupReference] resource_group_reference_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendServiceGroupHealth] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendServiceGroupHealth]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_backend_service_health(project, backend_service, resource_group_reference_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/backendServices/{backendService}/getHealth', options)
          command.request_representation = Google::Apis::ComputeV1::ResourceGroupReference::Representation
          command.request_object = resource_group_reference_object
          command.response_representation = Google::Apis::ComputeV1::BackendServiceGroupHealth::Representation
          command.response_class = Google::Apis::ComputeV1::BackendServiceGroupHealth
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a BackendService resource in the specified project using the data
        # included in the request. There are several restrictions and guidelines to keep
        # in mind when creating a backend service. Read  Restrictions and Guidelines for
        # more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_backend_service(project, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/backendServices', options)
          command.request_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of BackendService resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendServiceList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendServiceList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_backend_services(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/backendServices', options)
          command.response_representation = Google::Apis::ComputeV1::BackendServiceList::Representation
          command.response_class = Google::Apis::ComputeV1::BackendServiceList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patches the specified BackendService resource with the data included in the
        # request. There are several restrictions and guidelines to keep in mind when
        # updating a backend service. Read  Restrictions and Guidelines for more
        # information. This method supports PATCH semantics and uses the JSON merge
        # patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to patch.
        # @param [Google::Apis::ComputeV1::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_backend_service(project, backend_service, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/global/backendServices/{backendService}', options)
          command.request_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the security policy for the specified backend service.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to which the security policy should be set.
        #   The name should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::SecurityPolicyReference] security_policy_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_backend_service_security_policy(project, backend_service, security_policy_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/backendServices/{backendService}/setSecurityPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::SecurityPolicyReference::Representation
          command.request_object = security_policy_reference_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified BackendService resource with the data included in the
        # request. There are several restrictions and guidelines to keep in mind when
        # updating a backend service. Read  Restrictions and Guidelines for more
        # information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to update.
        # @param [Google::Apis::ComputeV1::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_backend_service(project, backend_service, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, '{project}/global/backendServices/{backendService}', options)
          command.request_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of disk types.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskTypeAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskTypeAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_disk_types(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/diskTypes', options)
          command.response_representation = Google::Apis::ComputeV1::DiskTypeAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::DiskTypeAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified disk type. Gets a list of available disk types by making
        # a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk_type
        #   Name of the disk type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_disk_type(project, zone, disk_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/diskTypes/{diskType}', options)
          command.response_representation = Google::Apis::ComputeV1::DiskType::Representation
          command.response_class = Google::Apis::ComputeV1::DiskType
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['diskType'] = disk_type unless disk_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of disk types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_disk_types(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/diskTypes', options)
          command.response_representation = Google::Apis::ComputeV1::DiskTypeList::Representation
          command.response_class = Google::Apis::ComputeV1::DiskTypeList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds existing resource policies to a disk. You can only add one policy which
        # will be applied to this disk for scheduling snapshot creation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   The disk name for this request.
        # @param [Google::Apis::ComputeV1::DisksAddResourcePoliciesRequest] disks_add_resource_policies_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_disk_resource_policies(project, zone, disk, disks_add_resource_policies_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/disks/{disk}/addResourcePolicies', options)
          command.request_representation = Google::Apis::ComputeV1::DisksAddResourcePoliciesRequest::Representation
          command.request_object = disks_add_resource_policies_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of persistent disks.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_disk(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/disks', options)
          command.response_representation = Google::Apis::ComputeV1::DiskAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::DiskAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a snapshot of a specified persistent disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   Name of the persistent disk to snapshot.
        # @param [Google::Apis::ComputeV1::Snapshot] snapshot_object
        # @param [Boolean] guest_flush
        #   [Input Only] Specifies to create an application consistent snapshot by
        #   informing the OS to prepare for the snapshot process. Currently only supported
        #   on Windows instances using the Volume Shadow Copy Service (VSS).
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_disk_snapshot(project, zone, disk, snapshot_object = nil, guest_flush: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/disks/{disk}/createSnapshot', options)
          command.request_representation = Google::Apis::ComputeV1::Snapshot::Representation
          command.request_object = snapshot_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['guestFlush'] = guest_flush unless guest_flush.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified persistent disk. Deleting a disk removes its data
        # permanently and is irreversible. However, deleting a disk does not delete any
        # snapshots previously made from the disk. You must separately delete snapshots.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   Name of the persistent disk to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_disk(project, zone, disk, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/zones/{zone}/disks/{disk}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns a specified persistent disk. Gets a list of available persistent disks
        # by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   Name of the persistent disk to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Disk] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Disk]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_disk(project, zone, disk, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/disks/{disk}', options)
          command.response_representation = Google::Apis::ComputeV1::Disk::Representation
          command.response_class = Google::Apis::ComputeV1::Disk
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_disk_iam_policy(project, zone, resource, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/disks/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a persistent disk in the specified project using the data in the
        # request. You can create a disk with a sourceImage, a sourceSnapshot, or create
        # an empty 500 GB data disk by omitting all properties. You can also create a
        # disk that is larger than the default size by specifying the sizeGb property.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Google::Apis::ComputeV1::Disk] disk_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] source_image
        #   Optional. Source image to restore onto a disk.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_disk(project, zone, disk_object = nil, request_id: nil, source_image: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/disks', options)
          command.request_representation = Google::Apis::ComputeV1::Disk::Representation
          command.request_object = disk_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['sourceImage'] = source_image unless source_image.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of persistent disks contained within the specified zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_disks(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/disks', options)
          command.response_representation = Google::Apis::ComputeV1::DiskList::Representation
          command.response_class = Google::Apis::ComputeV1::DiskList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Removes resource policies from a disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   The disk name for this request.
        # @param [Google::Apis::ComputeV1::DisksRemoveResourcePoliciesRequest] disks_remove_resource_policies_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_disk_resource_policies(project, zone, disk, disks_remove_resource_policies_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/disks/{disk}/removeResourcePolicies', options)
          command.request_representation = Google::Apis::ComputeV1::DisksRemoveResourcePoliciesRequest::Representation
          command.request_object = disks_remove_resource_policies_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Resizes the specified persistent disk. You can only increase the size of the
        # disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] disk
        #   The name of the persistent disk.
        # @param [Google::Apis::ComputeV1::DisksResizeRequest] disks_resize_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resize_disk(project, zone, disk, disks_resize_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/disks/{disk}/resize', options)
          command.request_representation = Google::Apis::ComputeV1::DisksResizeRequest::Representation
          command.request_object = disks_resize_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::ZoneSetPolicyRequest] zone_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_disk_iam_policy(project, zone, resource, zone_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/disks/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::ZoneSetPolicyRequest::Representation
          command.request_object = zone_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on a disk. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::ZoneSetLabelsRequest] zone_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_disk_labels(project, zone, resource, zone_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/disks/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::ZoneSetLabelsRequest::Representation
          command.request_object = zone_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_disk_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/disks/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified externalVpnGateway.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] external_vpn_gateway
        #   Name of the externalVpnGateways to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_external_vpn_gateway(project, external_vpn_gateway, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/externalVpnGateways/{externalVpnGateway}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['externalVpnGateway'] = external_vpn_gateway unless external_vpn_gateway.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified externalVpnGateway. Get a list of available
        # externalVpnGateways by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] external_vpn_gateway
        #   Name of the externalVpnGateway to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ExternalVpnGateway] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ExternalVpnGateway]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_external_vpn_gateway(project, external_vpn_gateway, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/externalVpnGateways/{externalVpnGateway}', options)
          command.response_representation = Google::Apis::ComputeV1::ExternalVpnGateway::Representation
          command.response_class = Google::Apis::ComputeV1::ExternalVpnGateway
          command.params['project'] = project unless project.nil?
          command.params['externalVpnGateway'] = external_vpn_gateway unless external_vpn_gateway.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a ExternalVpnGateway in the specified project using the data included
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::ExternalVpnGateway] external_vpn_gateway_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_external_vpn_gateway(project, external_vpn_gateway_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/externalVpnGateways', options)
          command.request_representation = Google::Apis::ComputeV1::ExternalVpnGateway::Representation
          command.request_object = external_vpn_gateway_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of ExternalVpnGateway available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ExternalVpnGatewayList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ExternalVpnGatewayList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_external_vpn_gateways(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/externalVpnGateways', options)
          command.response_representation = Google::Apis::ComputeV1::ExternalVpnGatewayList::Representation
          command.response_class = Google::Apis::ComputeV1::ExternalVpnGatewayList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on an ExternalVpnGateway. To learn more about labels, read the
        # Labeling Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetLabelsRequest] global_set_labels_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_external_vpn_gateway_labels(project, resource, global_set_labels_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/externalVpnGateways/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetLabelsRequest::Representation
          command.request_object = global_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_external_vpn_gateway_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/externalVpnGateways/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified firewall.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_firewall(project, firewall, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/firewalls/{firewall}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified firewall.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Firewall] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Firewall]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_firewall(project, firewall, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/firewalls/{firewall}', options)
          command.response_representation = Google::Apis::ComputeV1::Firewall::Representation
          command.response_class = Google::Apis::ComputeV1::Firewall
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a firewall rule in the specified project using the data included in
        # the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::Firewall] firewall_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_firewall(project, firewall_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/firewalls', options)
          command.request_representation = Google::Apis::ComputeV1::Firewall::Representation
          command.request_object = firewall_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of firewall rules available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::FirewallList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::FirewallList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_firewalls(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/firewalls', options)
          command.response_representation = Google::Apis::ComputeV1::FirewallList::Representation
          command.response_class = Google::Apis::ComputeV1::FirewallList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified firewall rule with the data included in the request.
        # This method supports PATCH semantics and uses the JSON merge patch format and
        # processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to patch.
        # @param [Google::Apis::ComputeV1::Firewall] firewall_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_firewall(project, firewall, firewall_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/global/firewalls/{firewall}', options)
          command.request_representation = Google::Apis::ComputeV1::Firewall::Representation
          command.request_object = firewall_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified firewall rule with the data included in the request.
        # Note that all fields will be updated if using PUT, even fields that are not
        # specified. To update individual fields, please use PATCH instead.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] firewall
        #   Name of the firewall rule to update.
        # @param [Google::Apis::ComputeV1::Firewall] firewall_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_firewall(project, firewall, firewall_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, '{project}/global/firewalls/{firewall}', options)
          command.request_representation = Google::Apis::ComputeV1::Firewall::Representation
          command.request_object = firewall_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['firewall'] = firewall unless firewall.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of forwarding rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ForwardingRuleAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ForwardingRuleAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_forwarding_rules(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/forwardingRules', options)
          command.response_representation = Google::Apis::ComputeV1::ForwardingRuleAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::ForwardingRuleAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified ForwardingRule resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_forwarding_rule(project, region, forwarding_rule, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/regions/{region}/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified ForwardingRule resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ForwardingRule] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ForwardingRule]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_forwarding_rule(project, region, forwarding_rule, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeV1::ForwardingRule::Representation
          command.response_class = Google::Apis::ComputeV1::ForwardingRule
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a ForwardingRule resource in the specified project and region using
        # the data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeV1::ForwardingRule] forwarding_rule_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_forwarding_rule(project, region, forwarding_rule_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/forwardingRules', options)
          command.request_representation = Google::Apis::ComputeV1::ForwardingRule::Representation
          command.request_object = forwarding_rule_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of ForwardingRule resources available to the specified
        # project and region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ForwardingRuleList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ForwardingRuleList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_forwarding_rules(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/forwardingRules', options)
          command.response_representation = Google::Apis::ComputeV1::ForwardingRuleList::Representation
          command.response_class = Google::Apis::ComputeV1::ForwardingRuleList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes target URL for forwarding rule. The new target should be of the same
        # type as the old target.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource in which target is to be set.
        # @param [Google::Apis::ComputeV1::TargetReference] target_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_forwarding_rule_target(project, region, forwarding_rule, target_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/forwardingRules/{forwardingRule}/setTarget', options)
          command.request_representation = Google::Apis::ComputeV1::TargetReference::Representation
          command.request_object = target_reference_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified address resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] address
        #   Name of the address resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_address(project, address, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['address'] = address unless address.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified address resource. Gets a list of available addresses by
        # making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] address
        #   Name of the address resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Address] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Address]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_address(project, address, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/addresses/{address}', options)
          command.response_representation = Google::Apis::ComputeV1::Address::Representation
          command.response_class = Google::Apis::ComputeV1::Address
          command.params['project'] = project unless project.nil?
          command.params['address'] = address unless address.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an address resource in the specified project by using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::Address] address_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_global_address(project, address_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/addresses', options)
          command.request_representation = Google::Apis::ComputeV1::Address::Representation
          command.request_object = address_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of global addresses.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::AddressList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::AddressList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_addresses(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/addresses', options)
          command.response_representation = Google::Apis::ComputeV1::AddressList::Representation
          command.response_class = Google::Apis::ComputeV1::AddressList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified GlobalForwardingRule resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_forwarding_rule(project, forwarding_rule, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified GlobalForwardingRule resource. Gets a list of available
        # forwarding rules by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ForwardingRule] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ForwardingRule]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_forwarding_rule(project, forwarding_rule, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/forwardingRules/{forwardingRule}', options)
          command.response_representation = Google::Apis::ComputeV1::ForwardingRule::Representation
          command.response_class = Google::Apis::ComputeV1::ForwardingRule
          command.params['project'] = project unless project.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a GlobalForwardingRule resource in the specified project using the
        # data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::ForwardingRule] forwarding_rule_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_global_forwarding_rule(project, forwarding_rule_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/forwardingRules', options)
          command.request_representation = Google::Apis::ComputeV1::ForwardingRule::Representation
          command.request_object = forwarding_rule_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of GlobalForwardingRule resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ForwardingRuleList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ForwardingRuleList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_forwarding_rules(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/forwardingRules', options)
          command.response_representation = Google::Apis::ComputeV1::ForwardingRuleList::Representation
          command.response_class = Google::Apis::ComputeV1::ForwardingRuleList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes target URL for the GlobalForwardingRule resource. The new target
        # should be of the same type as the old target.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] forwarding_rule
        #   Name of the ForwardingRule resource in which target is to be set.
        # @param [Google::Apis::ComputeV1::TargetReference] target_reference_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_global_forwarding_rule_target(project, forwarding_rule, target_reference_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/forwardingRules/{forwardingRule}/setTarget', options)
          command.request_representation = Google::Apis::ComputeV1::TargetReference::Representation
          command.request_object = target_reference_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['forwardingRule'] = forwarding_rule unless forwarding_rule.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of all operations.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::OperationAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::OperationAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_global_operation(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/operations', options)
          command.response_representation = Google::Apis::ComputeV1::OperationAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::OperationAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified Operations resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] operation
        #   Name of the Operations resource to delete.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [NilClass] No result returned for this method
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [void]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_global_operation(project, operation, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/operations/{operation}', options)
          command.params['project'] = project unless project.nil?
          command.params['operation'] = operation unless operation.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the specified Operations resource. Gets a list of operations by
        # making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] operation
        #   Name of the Operations resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_global_operation(project, operation, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/operations/{operation}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['operation'] = operation unless operation.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of Operation resources contained within the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::OperationList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::OperationList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_global_operations(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/operations', options)
          command.response_representation = Google::Apis::ComputeV1::OperationList::Representation
          command.response_class = Google::Apis::ComputeV1::OperationList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Waits for the specified Operations resource until it is done or timeout, and
        # retrieves the specified Operations resource. 1. Immediately returns when the
        # operation is already done. 2. Waits for no more than the default deadline (2
        # minutes, subject to change) and then returns the current state of the
        # operation, which may be DONE or still in progress. 3. Is best-effort: a. The
        # server can wait less than the default deadline or zero seconds, in overload
        # situations. b. There is no guarantee that the operation is actually done when
        # returns. 4. User should be prepared to retry if the operation is not DONE.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] operation
        #   Name of the Operations resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def wait_global_operation(project, operation, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/operations/{operation}/wait', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['operation'] = operation unless operation.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of all HealthCheck resources, regional and global,
        # available to the specified project.
        # @param [String] project
        #   Name of the project scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HealthChecksAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HealthChecksAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_health_check_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/healthChecks', options)
          command.response_representation = Google::Apis::ComputeV1::HealthChecksAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::HealthChecksAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified HealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_health_check(project, health_check, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/healthChecks/{healthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified HealthCheck resource. Gets a list of available health
        # checks by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HealthCheck] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HealthCheck]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_health_check(project, health_check, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/healthChecks/{healthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::HealthCheck::Representation
          command.response_class = Google::Apis::ComputeV1::HealthCheck
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a HealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::HealthCheck] health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_health_check(project, health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/healthChecks', options)
          command.request_representation = Google::Apis::ComputeV1::HealthCheck::Representation
          command.request_object = health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of HealthCheck resources available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HealthCheckList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HealthCheckList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_health_checks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/healthChecks', options)
          command.response_representation = Google::Apis::ComputeV1::HealthCheckList::Representation
          command.response_class = Google::Apis::ComputeV1::HealthCheckList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HealthCheck resource in the specified project using the data
        # included in the request. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to patch.
        # @param [Google::Apis::ComputeV1::HealthCheck] health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_health_check(project, health_check, health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/global/healthChecks/{healthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HealthCheck::Representation
          command.request_object = health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] health_check
        #   Name of the HealthCheck resource to update.
        # @param [Google::Apis::ComputeV1::HealthCheck] health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_health_check(project, health_check, health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, '{project}/global/healthChecks/{healthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HealthCheck::Representation
          command.request_object = health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['healthCheck'] = health_check unless health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified HttpHealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_http_health_check(project, http_health_check, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified HttpHealthCheck resource. Gets a list of available HTTP
        # health checks by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HttpHealthCheck] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HttpHealthCheck]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_http_health_check(project, http_health_check, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::HttpHealthCheck::Representation
          command.response_class = Google::Apis::ComputeV1::HttpHealthCheck
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a HttpHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::HttpHealthCheck] http_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_http_health_check(project, http_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/httpHealthChecks', options)
          command.request_representation = Google::Apis::ComputeV1::HttpHealthCheck::Representation
          command.request_object = http_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of HttpHealthCheck resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HttpHealthCheckList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HttpHealthCheckList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_http_health_checks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/httpHealthChecks', options)
          command.response_representation = Google::Apis::ComputeV1::HttpHealthCheckList::Representation
          command.response_class = Google::Apis::ComputeV1::HttpHealthCheckList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpHealthCheck resource in the specified project using the data
        # included in the request. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to patch.
        # @param [Google::Apis::ComputeV1::HttpHealthCheck] http_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_http_health_check(project, http_health_check, http_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HttpHealthCheck::Representation
          command.request_object = http_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] http_health_check
        #   Name of the HttpHealthCheck resource to update.
        # @param [Google::Apis::ComputeV1::HttpHealthCheck] http_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_http_health_check(project, http_health_check, http_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, '{project}/global/httpHealthChecks/{httpHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HttpHealthCheck::Representation
          command.request_object = http_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpHealthCheck'] = http_health_check unless http_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified HttpsHealthCheck resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_https_health_check(project, https_health_check, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified HttpsHealthCheck resource. Gets a list of available
        # HTTPS health checks by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HttpsHealthCheck] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HttpsHealthCheck]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_https_health_check(project, https_health_check, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.response_representation = Google::Apis::ComputeV1::HttpsHealthCheck::Representation
          command.response_class = Google::Apis::ComputeV1::HttpsHealthCheck
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a HttpsHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::HttpsHealthCheck] https_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_https_health_check(project, https_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/httpsHealthChecks', options)
          command.request_representation = Google::Apis::ComputeV1::HttpsHealthCheck::Representation
          command.request_object = https_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of HttpsHealthCheck resources available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::HttpsHealthCheckList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::HttpsHealthCheckList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_https_health_checks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/httpsHealthChecks', options)
          command.response_representation = Google::Apis::ComputeV1::HttpsHealthCheckList::Representation
          command.response_class = Google::Apis::ComputeV1::HttpsHealthCheckList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpsHealthCheck resource in the specified project using the data
        # included in the request. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to patch.
        # @param [Google::Apis::ComputeV1::HttpsHealthCheck] https_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_https_health_check(project, https_health_check, https_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HttpsHealthCheck::Representation
          command.request_object = https_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a HttpsHealthCheck resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] https_health_check
        #   Name of the HttpsHealthCheck resource to update.
        # @param [Google::Apis::ComputeV1::HttpsHealthCheck] https_health_check_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_https_health_check(project, https_health_check, https_health_check_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, '{project}/global/httpsHealthChecks/{httpsHealthCheck}', options)
          command.request_representation = Google::Apis::ComputeV1::HttpsHealthCheck::Representation
          command.request_object = https_health_check_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['httpsHealthCheck'] = https_health_check unless https_health_check.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified image.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] image
        #   Name of the image resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_image(project, image, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/images/{image}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['image'] = image unless image.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the deprecation status of an image.
        # If an empty request body is given, clears the deprecation status instead.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] image
        #   Image name.
        # @param [Google::Apis::ComputeV1::DeprecationStatus] deprecation_status_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def deprecate_image(project, image, deprecation_status_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/images/{image}/deprecate', options)
          command.request_representation = Google::Apis::ComputeV1::DeprecationStatus::Representation
          command.request_object = deprecation_status_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['image'] = image unless image.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified image. Gets a list of available images by making a list()
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] image
        #   Name of the image resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Image] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Image]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_image(project, image, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/images/{image}', options)
          command.response_representation = Google::Apis::ComputeV1::Image::Representation
          command.response_class = Google::Apis::ComputeV1::Image
          command.params['project'] = project unless project.nil?
          command.params['image'] = image unless image.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the latest image that is part of an image family and is not deprecated.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] family
        #   Name of the image family to search for.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Image] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Image]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_image_from_family(project, family, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/images/family/{family}', options)
          command.response_representation = Google::Apis::ComputeV1::Image::Representation
          command.response_class = Google::Apis::ComputeV1::Image
          command.params['project'] = project unless project.nil?
          command.params['family'] = family unless family.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_image_iam_policy(project, resource, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/images/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an image in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::Image] image_object
        # @param [Boolean] force_create
        #   Force image creation if true.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_image(project, image_object = nil, force_create: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/images', options)
          command.request_representation = Google::Apis::ComputeV1::Image::Representation
          command.request_object = image_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['forceCreate'] = force_create unless force_create.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of custom images available to the specified project. Custom
        # images are images you create that belong to your project. This method does not
        # get any images that belong to other projects, including publicly-available
        # images, like Debian 8. If you want to get a list of publicly-available images,
        # use this method to make a request to the respective image project, such as
        # debian-cloud or windows-cloud.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ImageList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ImageList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_images(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/images', options)
          command.response_representation = Google::Apis::ComputeV1::ImageList::Representation
          command.response_class = Google::Apis::ComputeV1::ImageList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetPolicyRequest] global_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_image_iam_policy(project, resource, global_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/images/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetPolicyRequest::Representation
          command.request_object = global_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on an image. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetLabelsRequest] global_set_labels_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_image_labels(project, resource, global_set_labels_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/images/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetLabelsRequest::Representation
          command.request_object = global_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_image_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/images/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Flags the specified instances to be removed from the managed instance group.
        # Abandoning an instance does not delete the instance, but it does remove the
        # instance from any target pools that are applied by the managed instance group.
        # This method reduces the targetSize of the managed instance group by the number
        # of instances that you abandon. This operation is marked as DONE when the
        # action is scheduled even if the instances have not yet been removed from the
        # group. You must separately verify the status of the abandoning action with the
        # listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # You can specify a maximum of 1000 instances with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersAbandonInstancesRequest] instance_group_managers_abandon_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def abandon_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_abandon_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/abandonInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersAbandonInstancesRequest::Representation
          command.request_object = instance_group_managers_abandon_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of managed instance groups and groups them by zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManagerAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManagerAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_instance_group_managers(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/instanceGroupManagers', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManagerAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManagerAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates instances with per-instance configs in this managed instance group.
        # Instances are created using the current instance template. The create
        # instances operation is marked DONE if the createInstances request is
        # successful. The underlying actions take additional time. You must separately
        # verify the status of the creating or actions with the listmanagedinstances
        # method.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located. It should
        #   conform to RFC1035.
        # @param [String] instance_group_manager
        #   The name of the managed instance group. It should conform to RFC1035.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersCreateInstancesRequest] instance_group_managers_create_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_create_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/createInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersCreateInstancesRequest::Representation
          command.request_object = instance_group_managers_create_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified managed instance group and all of the instances in that
        # group. Note that the instance group must not belong to a backend service. Read
        # Deleting an instance group for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group_manager(project, zone, instance_group_manager, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Flags the specified instances in the managed instance group for immediate
        # deletion. The instances are also removed from any target pools of which they
        # were a member. This method reduces the targetSize of the managed instance
        # group by the number of instances that you delete. This operation is marked as
        # DONE when the action is scheduled even if the instances are still being
        # deleted. You must separately verify the status of the deleting action with the
        # listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # You can specify a maximum of 1000 instances with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersDeleteInstancesRequest] instance_group_managers_delete_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_delete_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/deleteInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersDeleteInstancesRequest::Representation
          command.request_object = instance_group_managers_delete_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns all of the details about the specified managed instance group. Gets a
        # list of available managed instance groups by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManager] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManager]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_group_manager(project, zone, instance_group_manager, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManager::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManager
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a managed instance group using the information that you specify in the
        # request. After the group is created, instances in the group are created using
        # the specified instance template. This operation is marked as DONE when the
        # group is created even if the instances in the group have not yet been created.
        # You must separately verify the status of the individual instances with the
        # listmanagedinstances method.
        # A managed instance group can have up to 1000 VM instances per group. Please
        # contact Cloud Support if you need an increase in this limit.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManager] instance_group_manager_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance_group_manager(project, zone, instance_group_manager_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManager::Representation
          command.request_object = instance_group_manager_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of managed instance groups that are contained within the
        # specified project and zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManagerList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManagerList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_managers(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instanceGroupManagers', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManagerList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManagerList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all of the instances in the managed instance group. Each instance in the
        # list has a currentAction, which indicates the action that the managed instance
        # group is performing on the instance. For example, if the group is still
        # creating an instance, the currentAction is CREATING. If a previous action
        # failed, the list displays the errors for that failed action.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupManagersListManagedInstancesResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupManagersListManagedInstancesResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_manager_managed_instances(project, zone, instance_group_manager, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/listManagedInstances', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupManagersListManagedInstancesResponse::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupManagersListManagedInstancesResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['order_by'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates a managed instance group using the information that you specify in the
        # request. This operation is marked as DONE when the group is patched even if
        # the instances in the group are still in the process of being patched. You must
        # separately verify the status of the individual instances with the
        # listManagedInstances method. This method supports PATCH semantics and uses the
        # JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the managed instance group.
        # @param [String] instance_group_manager
        #   The name of the instance group manager.
        # @param [Google::Apis::ComputeV1::InstanceGroupManager] instance_group_manager_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_instance_group_manager(project, zone, instance_group_manager, instance_group_manager_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManager::Representation
          command.request_object = instance_group_manager_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Flags the specified instances in the managed instance group to be immediately
        # recreated. The instances are deleted and recreated using the current instance
        # template for the managed instance group. This operation is marked as DONE when
        # the flag is set even if the instances have not yet been recreated. You must
        # separately verify the status of the recreating action with the
        # listmanagedinstances method.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # You can specify a maximum of 1000 instances with this method per request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersRecreateInstancesRequest] instance_group_managers_recreate_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def recreate_instance_group_manager_instances(project, zone, instance_group_manager, instance_group_managers_recreate_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/recreateInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersRecreateInstancesRequest::Representation
          command.request_object = instance_group_managers_recreate_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Resizes the managed instance group. If you increase the size, the group
        # creates new instances using the current instance template. If you decrease the
        # size, the group deletes instances. The resize operation is marked DONE when
        # the resize actions are scheduled even if the group has not yet added or
        # deleted any instances. You must separately verify the status of the creating
        # or deleting actions with the listmanagedinstances method.
        # When resizing down, the instance group arbitrarily chooses the order in which
        # VMs are deleted. The group takes into account some VM attributes when making
        # the selection including:
        # + The status of the VM instance. + The health of the VM instance. + The
        # instance template version the VM is based on. + For regional managed instance
        # groups, the location of the VM instance.
        # This list is subject to change.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration has
        # elapsed before the VM instance is removed or deleted.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Fixnum] size
        #   The number of running instances that the managed instance group should
        #   maintain at any given time. The group automatically adds or removes instances
        #   to maintain the number of instances specified by this parameter.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resize_instance_group_manager(project, zone, instance_group_manager, size, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/resize', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['size'] = size unless size.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Specifies the instance template to use when creating new instances in this
        # group. The templates for existing instances in the group do not change unless
        # you recreate them.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersSetInstanceTemplateRequest] instance_group_managers_set_instance_template_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_group_manager_instance_template(project, zone, instance_group_manager, instance_group_managers_set_instance_template_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/setInstanceTemplate', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersSetInstanceTemplateRequest::Representation
          command.request_object = instance_group_managers_set_instance_template_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Modifies the target pools to which all instances in this managed instance
        # group are assigned. The target pools automatically apply to all of the
        # instances in the managed instance group. This operation is marked DONE when
        # you make the request even if the instances have not yet been added to their
        # target pools. The change might take some time to apply to all of the instances
        # in the group depending on the size of the group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the managed instance group is located.
        # @param [String] instance_group_manager
        #   The name of the managed instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroupManagersSetTargetPoolsRequest] instance_group_managers_set_target_pools_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_group_manager_target_pools(project, zone, instance_group_manager, instance_group_managers_set_target_pools_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroupManagers/{instanceGroupManager}/setTargetPools', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupManagersSetTargetPoolsRequest::Representation
          command.request_object = instance_group_managers_set_target_pools_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroupManager'] = instance_group_manager unless instance_group_manager.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds a list of instances to the specified instance group. All of the instances
        # in the instance group must be in the same network/subnetwork. Read  Adding
        # instances for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group where you are adding instances.
        # @param [Google::Apis::ComputeV1::InstanceGroupsAddInstancesRequest] instance_groups_add_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_instance_group_instances(project, zone, instance_group, instance_groups_add_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroups/{instanceGroup}/addInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupsAddInstancesRequest::Representation
          command.request_object = instance_groups_add_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of instance groups and sorts them by zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_instance_groups(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/instanceGroups', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified instance group. The instances in the group are not
        # deleted. Note that instance group must not belong to a backend service. Read
        # Deleting an instance group for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_group(project, zone, instance_group, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/zones/{zone}/instanceGroups/{instanceGroup}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified instance group. Gets a list of available instance groups
        # by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_group(project, zone, instance_group, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instanceGroups/{instanceGroup}', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroup::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroup
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an instance group in the specified project using the parameters that
        # are included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the instance group.
        # @param [Google::Apis::ComputeV1::InstanceGroup] instance_group_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance_group(project, zone, instance_group_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroups', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroup::Representation
          command.request_object = instance_group_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of instance groups that are located in the specified
        # project and zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_groups(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instanceGroups', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists the instances in the specified instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group from which you want to generate a list of
        #   included instances.
        # @param [Google::Apis::ComputeV1::InstanceGroupsListInstancesRequest] instance_groups_list_instances_request_object
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceGroupsListInstances] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceGroupsListInstances]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_group_instances(project, zone, instance_group, instance_groups_list_instances_request_object = nil, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroups/{instanceGroup}/listInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupsListInstancesRequest::Representation
          command.request_object = instance_groups_list_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::InstanceGroupsListInstances::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceGroupsListInstances
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Removes one or more instances from the specified instance group, but does not
        # delete those instances.
        # If the group is part of a backend service that has enabled connection draining,
        # it can take up to 60 seconds after the connection draining duration before
        # the VM instance is removed or deleted.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group where the specified instances will be removed.
        # @param [Google::Apis::ComputeV1::InstanceGroupsRemoveInstancesRequest] instance_groups_remove_instances_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_instance_group_instances(project, zone, instance_group, instance_groups_remove_instances_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroups/{instanceGroup}/removeInstances', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupsRemoveInstancesRequest::Representation
          command.request_object = instance_groups_remove_instances_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the named ports for the specified instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the instance group is located.
        # @param [String] instance_group
        #   The name of the instance group where the named ports are updated.
        # @param [Google::Apis::ComputeV1::InstanceGroupsSetNamedPortsRequest] instance_groups_set_named_ports_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_group_named_ports(project, zone, instance_group, instance_groups_set_named_ports_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instanceGroups/{instanceGroup}/setNamedPorts', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceGroupsSetNamedPortsRequest::Representation
          command.request_object = instance_groups_set_named_ports_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instanceGroup'] = instance_group unless instance_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified instance template. Deleting an instance template is
        # permanent and cannot be undone. It is not possible to delete templates that
        # are already in use by a managed instance group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] instance_template
        #   The name of the instance template to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_template(project, instance_template, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/instanceTemplates/{instanceTemplate}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['instanceTemplate'] = instance_template unless instance_template.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified instance template. Gets a list of available instance
        # templates by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] instance_template
        #   The name of the instance template.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceTemplate] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceTemplate]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_template(project, instance_template, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/instanceTemplates/{instanceTemplate}', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceTemplate::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceTemplate
          command.params['project'] = project unless project.nil?
          command.params['instanceTemplate'] = instance_template unless instance_template.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_template_iam_policy(project, resource, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/instanceTemplates/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an instance template in the specified project using the data that is
        # included in the request. If you are creating a new template to update an
        # existing instance group, your new instance template must use the same network
        # or, if applicable, the same subnetwork as the original template.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::InstanceTemplate] instance_template_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance_template(project, instance_template_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/instanceTemplates', options)
          command.request_representation = Google::Apis::ComputeV1::InstanceTemplate::Representation
          command.request_object = instance_template_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of instance templates that are contained within the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceTemplateList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceTemplateList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_templates(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/instanceTemplates', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceTemplateList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceTemplateList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetPolicyRequest] global_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_template_iam_policy(project, resource, global_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/instanceTemplates/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetPolicyRequest::Representation
          command.request_object = global_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_instance_template_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/instanceTemplates/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds an access config to an instance's network interface.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [String] network_interface
        #   The name of the network interface to add to this instance.
        # @param [Google::Apis::ComputeV1::AccessConfig] access_config_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_instance_access_config(project, zone, instance, network_interface, access_config_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/addAccessConfig', options)
          command.request_representation = Google::Apis::ComputeV1::AccessConfig::Representation
          command.request_object = access_config_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves aggregated list of all of the instances in your project across all
        # regions and zones.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_instances(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/instances', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Attaches an existing Disk resource to an instance. You must first create the
        # disk before you can attach it. It is not possible to create and attach a disk
        # at the same time. For more information, read Adding a persistent disk to your
        # instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [Google::Apis::ComputeV1::AttachedDisk] attached_disk_object
        # @param [Boolean] force_attach
        #   Whether to force attach the disk even if it's currently attached to another
        #   instance.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def attach_disk(project, zone, instance, attached_disk_object = nil, force_attach: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/attachDisk', options)
          command.request_representation = Google::Apis::ComputeV1::AttachedDisk::Representation
          command.request_object = attached_disk_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['forceAttach'] = force_attach unless force_attach.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified Instance resource. For more information, see Stopping or
        # Deleting an Instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/zones/{zone}/instances/{instance}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes an access config from an instance's network interface.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [String] access_config
        #   The name of the access config to delete.
        # @param [String] network_interface
        #   The name of the network interface.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_instance_access_config(project, zone, instance, access_config, network_interface, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/deleteAccessConfig', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['accessConfig'] = access_config unless access_config.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Detaches a disk from an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Instance name for this request.
        # @param [String] device_name
        #   The device name of the disk to detach. Make a get() request on the instance to
        #   view currently attached disks and device names.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def detach_disk(project, zone, instance, device_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/detachDisk', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['deviceName'] = device_name unless device_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified Instance resource. Gets a list of available instances by
        # making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Instance] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Instance]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance(project, zone, instance, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instances/{instance}', options)
          command.response_representation = Google::Apis::ComputeV1::Instance::Representation
          command.response_class = Google::Apis::ComputeV1::Instance
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified guest attributes entry.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] query_path
        #   Specifies the guest attributes path to be queried.
        # @param [String] variable_key
        #   Specifies the key for the guest attributes entry.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::GuestAttributes] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::GuestAttributes]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_guest_attributes(project, zone, instance, query_path: nil, variable_key: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instances/{instance}/getGuestAttributes', options)
          command.response_representation = Google::Apis::ComputeV1::GuestAttributes::Representation
          command.response_class = Google::Apis::ComputeV1::GuestAttributes
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['queryPath'] = query_path unless query_path.nil?
          command.query['variableKey'] = variable_key unless variable_key.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_iam_policy(project, zone, resource, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instances/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the last 1 MB of serial port output from the specified instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Fixnum] port
        #   Specifies which COM or serial port to retrieve data from.
        # @param [Fixnum] start
        #   Returns output starting from a specific byte position. Use this to page
        #   through output when the output is too large to return in a single request. For
        #   the initial request, leave this field unspecified. For subsequent calls, this
        #   field should be set to the next value returned in the previous call.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::SerialPortOutput] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::SerialPortOutput]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_serial_port_output(project, zone, instance, port: nil, start: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instances/{instance}/serialPort', options)
          command.response_representation = Google::Apis::ComputeV1::SerialPortOutput::Representation
          command.response_class = Google::Apis::ComputeV1::SerialPortOutput
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['port'] = port unless port.nil?
          command.query['start'] = start unless start.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the Shielded Instance Identity of an instance
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name or id of the instance scoping this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ShieldedInstanceIdentity] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ShieldedInstanceIdentity]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_instance_shielded_instance_identity(project, zone, instance, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instances/{instance}/getShieldedInstanceIdentity', options)
          command.response_representation = Google::Apis::ComputeV1::ShieldedInstanceIdentity::Representation
          command.response_class = Google::Apis::ComputeV1::ShieldedInstanceIdentity
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an instance resource in the specified project using the data included
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Google::Apis::ComputeV1::Instance] instance_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] source_instance_template
        #   Specifies instance template to create the instance.
        #   This field is optional. It can be a full or partial URL. For example, the
        #   following are all valid URLs to an instance template:
        #   - https://www.googleapis.com/compute/v1/projects/project/global/
        #   instanceTemplates/instanceTemplate
        #   - projects/project/global/instanceTemplates/instanceTemplate
        #   - global/instanceTemplates/instanceTemplate
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_instance(project, zone, instance_object = nil, request_id: nil, source_instance_template: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances', options)
          command.request_representation = Google::Apis::ComputeV1::Instance::Representation
          command.request_object = instance_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['sourceInstanceTemplate'] = source_instance_template unless source_instance_template.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of instances contained within the specified zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instances(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instances', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceList::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of referrers to instances contained within the specified
        # zone. For more information, read Viewing Referrers to VM Instances.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the target instance scoping this request, or '-' if the request should
        #   span over all instances in the container.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InstanceListReferrers] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InstanceListReferrers]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_instance_referrers(project, zone, instance, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/instances/{instance}/referrers', options)
          command.response_representation = Google::Apis::ComputeV1::InstanceListReferrers::Representation
          command.response_class = Google::Apis::ComputeV1::InstanceListReferrers
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Performs a reset on the instance. This is a hard reset the VM does not do a
        # graceful shutdown. For more information, see Resetting an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def reset_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/reset', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets deletion protection on the instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Boolean] deletion_protection
        #   Whether the resource should be protected against deletion.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_deletion_protection(project, zone, resource, deletion_protection: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{resource}/setDeletionProtection', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['deletionProtection'] = deletion_protection unless deletion_protection.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the auto-delete flag for a disk attached to an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [Boolean] auto_delete
        #   Whether to auto-delete the disk when the instance is deleted.
        # @param [String] device_name
        #   The device name of the disk to modify. Make a get() request on the instance to
        #   view currently attached disks and device names.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_disk_auto_delete(project, zone, instance, auto_delete, device_name, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setDiskAutoDelete', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['autoDelete'] = auto_delete unless auto_delete.nil?
          command.query['deviceName'] = device_name unless device_name.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::ZoneSetPolicyRequest] zone_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_iam_policy(project, zone, resource, zone_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::ZoneSetPolicyRequest::Representation
          command.request_object = zone_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets labels on an instance. To learn more about labels, read the Labeling
        # Resources documentation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::InstancesSetLabelsRequest] instances_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_labels(project, zone, instance, instances_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetLabelsRequest::Representation
          command.request_object = instances_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes the number and/or type of accelerator for a stopped instance to the
        # values specified in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::InstancesSetMachineResourcesRequest] instances_set_machine_resources_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_machine_resources(project, zone, instance, instances_set_machine_resources_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setMachineResources', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetMachineResourcesRequest::Representation
          command.request_object = instances_set_machine_resources_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes the machine type for a stopped instance to the machine type specified
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::InstancesSetMachineTypeRequest] instances_set_machine_type_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_machine_type(project, zone, instance, instances_set_machine_type_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setMachineType', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetMachineTypeRequest::Representation
          command.request_object = instances_set_machine_type_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets metadata for the specified instance to the data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::Metadata] metadata_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_metadata(project, zone, instance, metadata_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setMetadata', options)
          command.request_representation = Google::Apis::ComputeV1::Metadata::Representation
          command.request_object = metadata_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Changes the minimum CPU platform that this instance should use. This method
        # can only be called on a stopped instance. For more information, read
        # Specifying a Minimum CPU Platform.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::InstancesSetMinCpuPlatformRequest] instances_set_min_cpu_platform_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_min_cpu_platform(project, zone, instance, instances_set_min_cpu_platform_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setMinCpuPlatform', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetMinCpuPlatformRequest::Representation
          command.request_object = instances_set_min_cpu_platform_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets an instance's scheduling options.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Instance name for this request.
        # @param [Google::Apis::ComputeV1::Scheduling] scheduling_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_scheduling(project, zone, instance, scheduling_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setScheduling', options)
          command.request_representation = Google::Apis::ComputeV1::Scheduling::Representation
          command.request_object = scheduling_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the service account on the instance. For more information, read Changing
        # the service account and access scopes for an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to start.
        # @param [Google::Apis::ComputeV1::InstancesSetServiceAccountRequest] instances_set_service_account_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_service_account(project, zone, instance, instances_set_service_account_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setServiceAccount', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesSetServiceAccountRequest::Representation
          command.request_object = instances_set_service_account_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the Shielded Instance integrity policy for an instance. You can only use
        # this method on a running instance. This method supports PATCH semantics and
        # uses the JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name or id of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::ShieldedInstanceIntegrityPolicy] shielded_instance_integrity_policy_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_shielded_instance_integrity_policy(project, zone, instance, shielded_instance_integrity_policy_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/zones/{zone}/instances/{instance}/setShieldedInstanceIntegrityPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::ShieldedInstanceIntegrityPolicy::Representation
          command.request_object = shielded_instance_integrity_policy_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets network tags for the specified instance to the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::Tags] tags_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_instance_tags(project, zone, instance, tags_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/setTags', options)
          command.request_representation = Google::Apis::ComputeV1::Tags::Representation
          command.request_object = tags_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Simulates a maintenance event on the instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def simulate_instance_maintenance_event(project, zone, instance, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/simulateMaintenanceEvent', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Starts an instance that was stopped using the instances().stop method. For
        # more information, see Restart an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to start.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def start_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/start', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Starts an instance that was stopped using the instances().stop method. For
        # more information, see Restart an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to start.
        # @param [Google::Apis::ComputeV1::InstancesStartWithEncryptionKeyRequest] instances_start_with_encryption_key_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def start_instance_with_encryption_key(project, zone, instance, instances_start_with_encryption_key_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/startWithEncryptionKey', options)
          command.request_representation = Google::Apis::ComputeV1::InstancesStartWithEncryptionKeyRequest::Representation
          command.request_object = instances_start_with_encryption_key_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Stops a running instance, shutting it down cleanly, and allows you to restart
        # the instance at a later time. Stopped instances do not incur VM usage charges
        # while they are stopped. However, resources that the VM is using, such as
        # persistent disks and static IP addresses, will continue to be charged until
        # they are deleted. For more information, see Stopping an instance.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance resource to stop.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def stop_instance(project, zone, instance, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/stop', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_instance_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified access config from an instance's network interface with
        # the data included in the request. This method supports PATCH semantics and
        # uses the JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [String] network_interface
        #   The name of the network interface where the access config is attached.
        # @param [Google::Apis::ComputeV1::AccessConfig] access_config_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_instance_access_config(project, zone, instance, network_interface, access_config_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/instances/{instance}/updateAccessConfig', options)
          command.request_representation = Google::Apis::ComputeV1::AccessConfig::Representation
          command.request_object = access_config_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the Display config for a VM instance. You can only use this method on
        # a stopped VM instance. This method supports PATCH semantics and uses the JSON
        # merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::DisplayDevice] display_device_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_instance_display_device(project, zone, instance, display_device_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/zones/{zone}/instances/{instance}/updateDisplayDevice', options)
          command.request_representation = Google::Apis::ComputeV1::DisplayDevice::Representation
          command.request_object = display_device_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an instance's network interface. This method follows PATCH semantics.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   The instance name for this request.
        # @param [String] network_interface
        #   The name of the network interface to update.
        # @param [Google::Apis::ComputeV1::NetworkInterface] network_interface_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_instance_network_interface(project, zone, instance, network_interface, network_interface_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/zones/{zone}/instances/{instance}/updateNetworkInterface', options)
          command.request_representation = Google::Apis::ComputeV1::NetworkInterface::Representation
          command.request_object = network_interface_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['networkInterface'] = network_interface unless network_interface.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the Shielded Instance config for an instance. You can only use this
        # method on a stopped instance. This method supports PATCH semantics and uses
        # the JSON merge patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] instance
        #   Name or id of the instance scoping this request.
        # @param [Google::Apis::ComputeV1::ShieldedInstanceConfig] shielded_instance_config_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_instance_shielded_instance_config(project, zone, instance, shielded_instance_config_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/zones/{zone}/instances/{instance}/updateShieldedInstanceConfig', options)
          command.request_representation = Google::Apis::ComputeV1::ShieldedInstanceConfig::Representation
          command.request_object = shielded_instance_config_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['instance'] = instance unless instance.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of interconnect attachments.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InterconnectAttachmentAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InterconnectAttachmentAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_interconnect_attachment_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/interconnectAttachments', options)
          command.response_representation = Google::Apis::ComputeV1::InterconnectAttachmentAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::InterconnectAttachmentAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified interconnect attachment.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] interconnect_attachment
        #   Name of the interconnect attachment to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_interconnect_attachment(project, region, interconnect_attachment, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/regions/{region}/interconnectAttachments/{interconnectAttachment}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['interconnectAttachment'] = interconnect_attachment unless interconnect_attachment.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified interconnect attachment.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] interconnect_attachment
        #   Name of the interconnect attachment to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InterconnectAttachment] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InterconnectAttachment]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_interconnect_attachment(project, region, interconnect_attachment, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/interconnectAttachments/{interconnectAttachment}', options)
          command.response_representation = Google::Apis::ComputeV1::InterconnectAttachment::Representation
          command.response_class = Google::Apis::ComputeV1::InterconnectAttachment
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['interconnectAttachment'] = interconnect_attachment unless interconnect_attachment.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an InterconnectAttachment in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [Google::Apis::ComputeV1::InterconnectAttachment] interconnect_attachment_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [Boolean] validate_only
        #   If true, the request will not be committed.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_interconnect_attachment(project, region, interconnect_attachment_object = nil, request_id: nil, validate_only: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/interconnectAttachments', options)
          command.request_representation = Google::Apis::ComputeV1::InterconnectAttachment::Representation
          command.request_object = interconnect_attachment_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['validateOnly'] = validate_only unless validate_only.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of interconnect attachments contained within the specified
        # region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InterconnectAttachmentList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InterconnectAttachmentList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_interconnect_attachments(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/interconnectAttachments', options)
          command.response_representation = Google::Apis::ComputeV1::InterconnectAttachmentList::Representation
          command.response_class = Google::Apis::ComputeV1::InterconnectAttachmentList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified interconnect attachment with the data included in the
        # request. This method supports PATCH semantics and uses the JSON merge patch
        # format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] interconnect_attachment
        #   Name of the interconnect attachment to patch.
        # @param [Google::Apis::ComputeV1::InterconnectAttachment] interconnect_attachment_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_interconnect_attachment(project, region, interconnect_attachment, interconnect_attachment_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/regions/{region}/interconnectAttachments/{interconnectAttachment}', options)
          command.request_representation = Google::Apis::ComputeV1::InterconnectAttachment::Representation
          command.request_object = interconnect_attachment_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['interconnectAttachment'] = interconnect_attachment unless interconnect_attachment.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the details for the specified interconnect location. Gets a list of
        # available interconnect locations by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] interconnect_location
        #   Name of the interconnect location to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InterconnectLocation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InterconnectLocation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_interconnect_location(project, interconnect_location, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/interconnectLocations/{interconnectLocation}', options)
          command.response_representation = Google::Apis::ComputeV1::InterconnectLocation::Representation
          command.response_class = Google::Apis::ComputeV1::InterconnectLocation
          command.params['project'] = project unless project.nil?
          command.params['interconnectLocation'] = interconnect_location unless interconnect_location.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of interconnect locations available to the specified
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InterconnectLocationList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InterconnectLocationList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_interconnect_locations(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/interconnectLocations', options)
          command.response_representation = Google::Apis::ComputeV1::InterconnectLocationList::Representation
          command.response_class = Google::Apis::ComputeV1::InterconnectLocationList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified interconnect.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] interconnect
        #   Name of the interconnect to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_interconnect(project, interconnect, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/interconnects/{interconnect}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['interconnect'] = interconnect unless interconnect.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified interconnect. Get a list of available interconnects by
        # making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] interconnect
        #   Name of the interconnect to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Interconnect] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Interconnect]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_interconnect(project, interconnect, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/interconnects/{interconnect}', options)
          command.response_representation = Google::Apis::ComputeV1::Interconnect::Representation
          command.response_class = Google::Apis::ComputeV1::Interconnect
          command.params['project'] = project unless project.nil?
          command.params['interconnect'] = interconnect unless interconnect.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the interconnectDiagnostics for the specified interconnect.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] interconnect
        #   Name of the interconnect resource to query.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InterconnectsGetDiagnosticsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InterconnectsGetDiagnosticsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_interconnect_diagnostics(project, interconnect, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/interconnects/{interconnect}/getDiagnostics', options)
          command.response_representation = Google::Apis::ComputeV1::InterconnectsGetDiagnosticsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::InterconnectsGetDiagnosticsResponse
          command.params['project'] = project unless project.nil?
          command.params['interconnect'] = interconnect unless interconnect.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a Interconnect in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::Interconnect] interconnect_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_interconnect(project, interconnect_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/interconnects', options)
          command.request_representation = Google::Apis::ComputeV1::Interconnect::Representation
          command.request_object = interconnect_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of interconnect available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::InterconnectList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::InterconnectList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_interconnects(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/interconnects', options)
          command.response_representation = Google::Apis::ComputeV1::InterconnectList::Representation
          command.response_class = Google::Apis::ComputeV1::InterconnectList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified interconnect with the data included in the request. This
        # method supports PATCH semantics and uses the JSON merge patch format and
        # processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] interconnect
        #   Name of the interconnect to update.
        # @param [Google::Apis::ComputeV1::Interconnect] interconnect_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_interconnect(project, interconnect, interconnect_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/global/interconnects/{interconnect}', options)
          command.request_representation = Google::Apis::ComputeV1::Interconnect::Representation
          command.request_object = interconnect_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['interconnect'] = interconnect unless interconnect.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Return a specified license code. License codes are mirrored across all
        # projects that have permissions to read the License Code.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] license_code
        #   Number corresponding to the License code resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::LicenseCode] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::LicenseCode]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_license_code(project, license_code, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/licenseCodes/{licenseCode}', options)
          command.response_representation = Google::Apis::ComputeV1::LicenseCode::Representation
          command.response_class = Google::Apis::ComputeV1::LicenseCode
          command.params['project'] = project unless project.nil?
          command.params['licenseCode'] = license_code unless license_code.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_license_code_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/licenseCodes/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified license.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] license
        #   Name of the license resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_license(project, license, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/licenses/{license}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['license'] = license unless license.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified License resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] license
        #   Name of the License resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::License] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::License]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_license(project, license, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/licenses/{license}', options)
          command.response_representation = Google::Apis::ComputeV1::License::Representation
          command.response_class = Google::Apis::ComputeV1::License
          command.params['project'] = project unless project.nil?
          command.params['license'] = license unless license.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_license_iam_policy(project, resource, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/licenses/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Create a License resource in the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::License] license_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_license(project, license_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/licenses', options)
          command.request_representation = Google::Apis::ComputeV1::License::Representation
          command.request_object = license_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of licenses available in the specified project. This method
        # does not get any licenses that belong to other projects, including licenses
        # attached to publicly-available images, like Debian 9. If you want to get a
        # list of publicly-available licenses, use this method to make a request to the
        # respective image project, such as debian-cloud or windows-cloud.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::LicensesListResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::LicensesListResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_licenses(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/licenses', options)
          command.response_representation = Google::Apis::ComputeV1::LicensesListResponse::Representation
          command.response_class = Google::Apis::ComputeV1::LicensesListResponse
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::GlobalSetPolicyRequest] global_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_license_iam_policy(project, resource, global_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/licenses/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::GlobalSetPolicyRequest::Representation
          command.request_object = global_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_license_iam_permissions(project, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/licenses/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of machine types.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::MachineTypeAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::MachineTypeAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_aggregated_machine_types(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/machineTypes', options)
          command.response_representation = Google::Apis::ComputeV1::MachineTypeAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::MachineTypeAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified machine type. Gets a list of available machine types by
        # making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] machine_type
        #   Name of the machine type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::MachineType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::MachineType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_machine_type(project, zone, machine_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/machineTypes/{machineType}', options)
          command.response_representation = Google::Apis::ComputeV1::MachineType::Representation
          command.response_class = Google::Apis::ComputeV1::MachineType
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['machineType'] = machine_type unless machine_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of machine types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::MachineTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::MachineTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_machine_types(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/machineTypes', options)
          command.response_representation = Google::Apis::ComputeV1::MachineTypeList::Representation
          command.response_class = Google::Apis::ComputeV1::MachineTypeList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of network endpoint groups and sorts them by zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NetworkEndpointGroupAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NetworkEndpointGroupAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_network_endpoint_group_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/networkEndpointGroups', options)
          command.response_representation = Google::Apis::ComputeV1::NetworkEndpointGroupAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::NetworkEndpointGroupAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Attach a list of network endpoints to the specified network endpoint group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group where you are attaching network
        #   endpoints to. It should comply with RFC1035.
        # @param [Google::Apis::ComputeV1::NetworkEndpointGroupsAttachEndpointsRequest] network_endpoint_groups_attach_endpoints_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def attach_network_endpoint_group_network_endpoints(project, zone, network_endpoint_group, network_endpoint_groups_attach_endpoints_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/networkEndpointGroups/{networkEndpointGroup}/attachNetworkEndpoints', options)
          command.request_representation = Google::Apis::ComputeV1::NetworkEndpointGroupsAttachEndpointsRequest::Representation
          command.request_object = network_endpoint_groups_attach_endpoints_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified network endpoint group. The network endpoints in the NEG
        # and the VM instances they belong to are not terminated when the NEG is deleted.
        # Note that the NEG cannot be deleted if there are backend services referencing
        # it.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group to delete. It should comply with
        #   RFC1035.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_network_endpoint_group(project, zone, network_endpoint_group, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/zones/{zone}/networkEndpointGroups/{networkEndpointGroup}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Detach a list of network endpoints from the specified network endpoint group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group where you are removing network
        #   endpoints. It should comply with RFC1035.
        # @param [Google::Apis::ComputeV1::NetworkEndpointGroupsDetachEndpointsRequest] network_endpoint_groups_detach_endpoints_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def detach_network_endpoint_group_network_endpoints(project, zone, network_endpoint_group, network_endpoint_groups_detach_endpoints_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/networkEndpointGroups/{networkEndpointGroup}/detachNetworkEndpoints', options)
          command.request_representation = Google::Apis::ComputeV1::NetworkEndpointGroupsDetachEndpointsRequest::Representation
          command.request_object = network_endpoint_groups_detach_endpoints_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified network endpoint group. Gets a list of available network
        # endpoint groups by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group. It should comply with RFC1035.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NetworkEndpointGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NetworkEndpointGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_network_endpoint_group(project, zone, network_endpoint_group, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/networkEndpointGroups/{networkEndpointGroup}', options)
          command.response_representation = Google::Apis::ComputeV1::NetworkEndpointGroup::Representation
          command.response_class = Google::Apis::ComputeV1::NetworkEndpointGroup
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a network endpoint group in the specified project using the parameters
        # that are included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where you want to create the network endpoint group. It
        #   should comply with RFC1035.
        # @param [Google::Apis::ComputeV1::NetworkEndpointGroup] network_endpoint_group_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_network_endpoint_group(project, zone, network_endpoint_group_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/networkEndpointGroups', options)
          command.request_representation = Google::Apis::ComputeV1::NetworkEndpointGroup::Representation
          command.request_object = network_endpoint_group_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of network endpoint groups that are located in the
        # specified project and zone.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NetworkEndpointGroupList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NetworkEndpointGroupList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_network_endpoint_groups(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/networkEndpointGroups', options)
          command.response_representation = Google::Apis::ComputeV1::NetworkEndpointGroupList::Representation
          command.response_class = Google::Apis::ComputeV1::NetworkEndpointGroupList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists the network endpoints in the specified network endpoint group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone where the network endpoint group is located. It should
        #   comply with RFC1035.
        # @param [String] network_endpoint_group
        #   The name of the network endpoint group from which you want to generate a list
        #   of included network endpoints. It should comply with RFC1035.
        # @param [Google::Apis::ComputeV1::NetworkEndpointGroupsListEndpointsRequest] network_endpoint_groups_list_endpoints_request_object
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NetworkEndpointGroupsListNetworkEndpoints] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NetworkEndpointGroupsListNetworkEndpoints]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_network_endpoint_group_network_endpoints(project, zone, network_endpoint_group, network_endpoint_groups_list_endpoints_request_object = nil, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/networkEndpointGroups/{networkEndpointGroup}/listNetworkEndpoints', options)
          command.request_representation = Google::Apis::ComputeV1::NetworkEndpointGroupsListEndpointsRequest::Representation
          command.request_object = network_endpoint_groups_list_endpoints_request_object
          command.response_representation = Google::Apis::ComputeV1::NetworkEndpointGroupsListNetworkEndpoints::Representation
          command.response_class = Google::Apis::ComputeV1::NetworkEndpointGroupsListNetworkEndpoints
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['networkEndpointGroup'] = network_endpoint_group unless network_endpoint_group.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_network_endpoint_group_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/networkEndpointGroups/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds a peering to the specified network.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network resource to add peering to.
        # @param [Google::Apis::ComputeV1::NetworksAddPeeringRequest] networks_add_peering_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_network_peering(project, network, networks_add_peering_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/networks/{network}/addPeering', options)
          command.request_representation = Google::Apis::ComputeV1::NetworksAddPeeringRequest::Representation
          command.request_object = networks_add_peering_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified network.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_network(project, network, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/global/networks/{network}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified network. Gets a list of available networks by making a
        # list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Network] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Network]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_network(project, network, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/networks/{network}', options)
          command.response_representation = Google::Apis::ComputeV1::Network::Representation
          command.response_class = Google::Apis::ComputeV1::Network
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a network in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::Network] network_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_network(project, network_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/networks', options)
          command.request_representation = Google::Apis::ComputeV1::Network::Representation
          command.request_object = network_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of networks available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NetworkList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NetworkList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_networks(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/networks', options)
          command.response_representation = Google::Apis::ComputeV1::NetworkList::Representation
          command.response_class = Google::Apis::ComputeV1::NetworkList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists the peering routes exchanged over peering connection.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network for this request.
        # @param [String] direction
        #   The direction of the exchanged routes.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] peering_name
        #   The response will show routes exchanged over the given peering connection.
        # @param [String] region
        #   The region of the request. The response will include all subnet routes, static
        #   routes and dynamic routes in the region.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ExchangedPeeringRoutesList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ExchangedPeeringRoutesList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_network_peering_routes(project, network, direction: nil, filter: nil, max_results: nil, order_by: nil, page_token: nil, peering_name: nil, region: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/global/networks/{network}/listPeeringRoutes', options)
          command.response_representation = Google::Apis::ComputeV1::ExchangedPeeringRoutesList::Representation
          command.response_class = Google::Apis::ComputeV1::ExchangedPeeringRoutesList
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['direction'] = direction unless direction.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['peeringName'] = peering_name unless peering_name.nil?
          command.query['region'] = region unless region.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Patches the specified network with the data included in the request. Only the
        # following fields can be modified: routingConfig.routingMode.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network to update.
        # @param [Google::Apis::ComputeV1::Network] network_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_network(project, network, network_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/global/networks/{network}', options)
          command.request_representation = Google::Apis::ComputeV1::Network::Representation
          command.request_object = network_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Removes a peering from the specified network.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network resource to remove peering from.
        # @param [Google::Apis::ComputeV1::NetworksRemovePeeringRequest] networks_remove_peering_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_network_peering(project, network, networks_remove_peering_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/networks/{network}/removePeering', options)
          command.request_representation = Google::Apis::ComputeV1::NetworksRemovePeeringRequest::Representation
          command.request_object = networks_remove_peering_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Switches the network mode from auto subnet mode to custom subnet mode.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network to be updated.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def switch_network_to_custom_mode(project, network, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/global/networks/{network}/switchToCustomMode', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified network peering with the data included in the request
        # Only the following fields can be modified: NetworkPeering.export_custom_routes,
        # and NetworkPeering.import_custom_routes
        # @param [String] project
        #   Project ID for this request.
        # @param [String] network
        #   Name of the network resource which the updated peering is belonging to.
        # @param [Google::Apis::ComputeV1::NetworksUpdatePeeringRequest] networks_update_peering_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_network_peering(project, network, networks_update_peering_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/global/networks/{network}/updatePeering', options)
          command.request_representation = Google::Apis::ComputeV1::NetworksUpdatePeeringRequest::Representation
          command.request_object = networks_update_peering_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['network'] = network unless network.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds specified number of nodes to the node group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the NodeGroup resource.
        # @param [Google::Apis::ComputeV1::NodeGroupsAddNodesRequest] node_groups_add_nodes_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_node_group_nodes(project, zone, node_group, node_groups_add_nodes_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/nodeGroups/{nodeGroup}/addNodes', options)
          command.request_representation = Google::Apis::ComputeV1::NodeGroupsAddNodesRequest::Representation
          command.request_object = node_groups_add_nodes_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of node groups. Note: use nodeGroups.listNodes
        # for more details about each group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NodeGroupAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NodeGroupAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_node_group_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/nodeGroups', options)
          command.response_representation = Google::Apis::ComputeV1::NodeGroupAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::NodeGroupAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified NodeGroup resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the NodeGroup resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_node_group(project, zone, node_group, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/zones/{zone}/nodeGroups/{nodeGroup}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes specified nodes from the node group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the NodeGroup resource whose nodes will be deleted.
        # @param [Google::Apis::ComputeV1::NodeGroupsDeleteNodesRequest] node_groups_delete_nodes_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_node_group_nodes(project, zone, node_group, node_groups_delete_nodes_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/nodeGroups/{nodeGroup}/deleteNodes', options)
          command.request_representation = Google::Apis::ComputeV1::NodeGroupsDeleteNodesRequest::Representation
          command.request_object = node_groups_delete_nodes_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified NodeGroup. Get a list of available NodeGroups by making
        # a list() request. Note: the "nodes" field should not be used. Use nodeGroups.
        # listNodes instead.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the node group to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NodeGroup] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NodeGroup]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_node_group(project, zone, node_group, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/nodeGroups/{nodeGroup}', options)
          command.response_representation = Google::Apis::ComputeV1::NodeGroup::Representation
          command.response_class = Google::Apis::ComputeV1::NodeGroup
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_node_group_iam_policy(project, zone, resource, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/nodeGroups/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a NodeGroup resource in the specified project using the data included
        # in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [Fixnum] initial_node_count
        #   Initial count of nodes in the node group.
        # @param [Google::Apis::ComputeV1::NodeGroup] node_group_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_node_group(project, zone, initial_node_count, node_group_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/nodeGroups', options)
          command.request_representation = Google::Apis::ComputeV1::NodeGroup::Representation
          command.request_object = node_group_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['initialNodeCount'] = initial_node_count unless initial_node_count.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of node groups available to the specified project. Note: use
        # nodeGroups.listNodes for more details about each group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NodeGroupList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NodeGroupList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_node_groups(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/nodeGroups', options)
          command.response_representation = Google::Apis::ComputeV1::NodeGroupList::Representation
          command.response_class = Google::Apis::ComputeV1::NodeGroupList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists nodes in the node group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the NodeGroup resource whose nodes you want to list.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NodeGroupsListNodes] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NodeGroupsListNodes]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_node_group_nodes(project, zone, node_group, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/nodeGroups/{nodeGroup}/listNodes', options)
          command.response_representation = Google::Apis::ComputeV1::NodeGroupsListNodes::Representation
          command.response_class = Google::Apis::ComputeV1::NodeGroupsListNodes
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::ZoneSetPolicyRequest] zone_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_node_group_iam_policy(project, zone, resource, zone_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/nodeGroups/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::ZoneSetPolicyRequest::Representation
          command.request_object = zone_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the node template of the node group.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_group
        #   Name of the NodeGroup resource to update.
        # @param [Google::Apis::ComputeV1::NodeGroupsSetNodeTemplateRequest] node_groups_set_node_template_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_node_group_node_template(project, zone, node_group, node_groups_set_node_template_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/nodeGroups/{nodeGroup}/setNodeTemplate', options)
          command.request_representation = Google::Apis::ComputeV1::NodeGroupsSetNodeTemplateRequest::Representation
          command.request_object = node_groups_set_node_template_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeGroup'] = node_group unless node_group.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_node_group_iam_permissions(project, zone, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/zones/{zone}/nodeGroups/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of node templates.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NodeTemplateAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NodeTemplateAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_node_template_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/nodeTemplates', options)
          command.response_representation = Google::Apis::ComputeV1::NodeTemplateAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::NodeTemplateAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified NodeTemplate resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] node_template
        #   Name of the NodeTemplate resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_node_template(project, region, node_template, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/regions/{region}/nodeTemplates/{nodeTemplate}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['nodeTemplate'] = node_template unless node_template.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified node template. Gets a list of available node templates
        # by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] node_template
        #   Name of the node template to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NodeTemplate] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NodeTemplate]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_node_template(project, region, node_template, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/nodeTemplates/{nodeTemplate}', options)
          command.response_representation = Google::Apis::ComputeV1::NodeTemplate::Representation
          command.response_class = Google::Apis::ComputeV1::NodeTemplate
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['nodeTemplate'] = node_template unless node_template.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the access control policy for a resource. May be empty if no such policy
        # or resource exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_node_template_iam_policy(project, region, resource, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/nodeTemplates/{resource}/getIamPolicy', options)
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a NodeTemplate resource in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [Google::Apis::ComputeV1::NodeTemplate] node_template_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_node_template(project, region, node_template_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/nodeTemplates', options)
          command.request_representation = Google::Apis::ComputeV1::NodeTemplate::Representation
          command.request_object = node_template_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of node templates available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NodeTemplateList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NodeTemplateList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_node_templates(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/nodeTemplates', options)
          command.response_representation = Google::Apis::ComputeV1::NodeTemplateList::Representation
          command.response_class = Google::Apis::ComputeV1::NodeTemplateList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the access control policy on the specified resource. Replaces any
        # existing policy.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::RegionSetPolicyRequest] region_set_policy_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Policy] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Policy]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_node_template_iam_policy(project, region, resource, region_set_policy_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/nodeTemplates/{resource}/setIamPolicy', options)
          command.request_representation = Google::Apis::ComputeV1::RegionSetPolicyRequest::Representation
          command.request_object = region_set_policy_request_object
          command.response_representation = Google::Apis::ComputeV1::Policy::Representation
          command.response_class = Google::Apis::ComputeV1::Policy
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_node_template_iam_permissions(project, region, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/nodeTemplates/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of node types.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NodeTypeAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NodeTypeAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_node_type_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/nodeTypes', options)
          command.response_representation = Google::Apis::ComputeV1::NodeTypeAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::NodeTypeAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified node type. Gets a list of available node types by making
        # a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] node_type
        #   Name of the node type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NodeType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NodeType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_node_type(project, zone, node_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/nodeTypes/{nodeType}', options)
          command.response_representation = Google::Apis::ComputeV1::NodeType::Representation
          command.response_class = Google::Apis::ComputeV1::NodeType
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.params['nodeType'] = node_type unless node_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of node types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] zone
        #   The name of the zone for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::NodeTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::NodeTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_node_types(project, zone, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/zones/{zone}/nodeTypes', options)
          command.response_representation = Google::Apis::ComputeV1::NodeTypeList::Representation
          command.response_class = Google::Apis::ComputeV1::NodeTypeList
          command.params['project'] = project unless project.nil?
          command.params['zone'] = zone unless zone.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Disable this project as a shared VPC host project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def disable_project_xpn_host(project, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/disableXpnHost', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Disable a service resource (also known as service project) associated with
        # this host project.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::ProjectsDisableXpnResourceRequest] projects_disable_xpn_resource_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def disable_project_xpn_resource(project, projects_disable_xpn_resource_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/disableXpnResource', options)
          command.request_representation = Google::Apis::ComputeV1::ProjectsDisableXpnResourceRequest::Representation
          command.request_object = projects_disable_xpn_resource_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Enable this project as a shared VPC host project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def enable_project_xpn_host(project, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/enableXpnHost', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Enable service resource (a.k.a service project) for a host project, so that
        # subnets in the host project can be used by instances in the service project.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::ProjectsEnableXpnResourceRequest] projects_enable_xpn_resource_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def enable_project_xpn_resource(project, projects_enable_xpn_resource_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/enableXpnResource', options)
          command.request_representation = Google::Apis::ComputeV1::ProjectsEnableXpnResourceRequest::Representation
          command.request_object = projects_enable_xpn_resource_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified Project resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Project] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Project]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project(project, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}', options)
          command.response_representation = Google::Apis::ComputeV1::Project::Representation
          command.response_class = Google::Apis::ComputeV1::Project
          command.params['project'] = project unless project.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the shared VPC host project that this project links to. May be empty if
        # no link exists.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Project] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Project]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_xpn_host(project, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/getXpnHost', options)
          command.response_representation = Google::Apis::ComputeV1::Project::Representation
          command.response_class = Google::Apis::ComputeV1::Project
          command.params['project'] = project unless project.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets service resources (a.k.a service project) associated with this host
        # project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::ProjectsGetXpnResources] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::ProjectsGetXpnResources]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_project_xpn_resources(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/getXpnResources', options)
          command.response_representation = Google::Apis::ComputeV1::ProjectsGetXpnResources::Representation
          command.response_class = Google::Apis::ComputeV1::ProjectsGetXpnResources
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['order_by'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Lists all shared VPC host projects visible to the user in an organization.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::ProjectsListXpnHostsRequest] projects_list_xpn_hosts_request_object
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::XpnHostList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::XpnHostList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_project_xpn_hosts(project, projects_list_xpn_hosts_request_object = nil, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/listXpnHosts', options)
          command.request_representation = Google::Apis::ComputeV1::ProjectsListXpnHostsRequest::Representation
          command.request_object = projects_list_xpn_hosts_request_object
          command.response_representation = Google::Apis::ComputeV1::XpnHostList::Representation
          command.response_class = Google::Apis::ComputeV1::XpnHostList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['order_by'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Moves a persistent disk from one zone to another.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::MoveDiskRequest] move_disk_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def move_disk(project, move_disk_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/moveDisk', options)
          command.request_representation = Google::Apis::ComputeV1::MoveDiskRequest::Representation
          command.request_object = move_disk_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Moves an instance and its attached persistent disks from one zone to another.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::MoveInstanceRequest] move_instance_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def move_instance(project, move_instance_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/moveInstance', options)
          command.request_representation = Google::Apis::ComputeV1::MoveInstanceRequest::Representation
          command.request_object = move_instance_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets metadata common to all instances within the specified project using the
        # data included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::Metadata] metadata_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_common_instance_metadata(project, metadata_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/setCommonInstanceMetadata', options)
          command.request_representation = Google::Apis::ComputeV1::Metadata::Representation
          command.request_object = metadata_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the default network tier of the project. The default network tier is used
        # when an address/forwardingRule/instance is created without specifying the
        # network tier field.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::ProjectsSetDefaultNetworkTierRequest] projects_set_default_network_tier_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_project_default_network_tier(project, projects_set_default_network_tier_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/setDefaultNetworkTier', options)
          command.request_representation = Google::Apis::ComputeV1::ProjectsSetDefaultNetworkTierRequest::Representation
          command.request_object = projects_set_default_network_tier_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Enables the usage export feature and sets the usage export bucket where
        # reports are stored. If you provide an empty request body using this method,
        # the usage export feature will be disabled.
        # @param [String] project
        #   Project ID for this request.
        # @param [Google::Apis::ComputeV1::UsageExportLocation] usage_export_location_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_usage_export_bucket(project, usage_export_location_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/setUsageExportBucket', options)
          command.request_representation = Google::Apis::ComputeV1::UsageExportLocation::Representation
          command.request_object = usage_export_location_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified autoscaler.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] autoscaler
        #   Name of the autoscaler to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_region_autoscaler(project, region, autoscaler, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/regions/{region}/autoscalers/{autoscaler}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified autoscaler.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] autoscaler
        #   Name of the autoscaler to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Autoscaler] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Autoscaler]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_autoscaler(project, region, autoscaler, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/autoscalers/{autoscaler}', options)
          command.response_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.response_class = Google::Apis::ComputeV1::Autoscaler
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates an autoscaler in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeV1::Autoscaler] autoscaler_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_region_autoscaler(project, region, autoscaler_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of autoscalers contained within the specified region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::RegionAutoscalerList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::RegionAutoscalerList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_autoscalers(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/autoscalers', options)
          command.response_representation = Google::Apis::ComputeV1::RegionAutoscalerList::Representation
          command.response_class = Google::Apis::ComputeV1::RegionAutoscalerList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an autoscaler in the specified project using the data included in the
        # request. This method supports PATCH semantics and uses the JSON merge patch
        # format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeV1::Autoscaler] autoscaler_object
        # @param [String] autoscaler
        #   Name of the autoscaler to patch.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_region_autoscaler(project, region, autoscaler_object = nil, autoscaler: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/regions/{region}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates an autoscaler in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeV1::Autoscaler] autoscaler_object
        # @param [String] autoscaler
        #   Name of the autoscaler to update.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_region_autoscaler(project, region, autoscaler_object = nil, autoscaler: nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, '{project}/regions/{region}/autoscalers', options)
          command.request_representation = Google::Apis::ComputeV1::Autoscaler::Representation
          command.request_object = autoscaler_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['autoscaler'] = autoscaler unless autoscaler.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified regional BackendService resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_region_backend_service(project, region, backend_service, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/regions/{region}/backendServices/{backendService}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified regional BackendService resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendService] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendService]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_backend_service(project, region, backend_service, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/backendServices/{backendService}', options)
          command.response_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.response_class = Google::Apis::ComputeV1::BackendService
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Gets the most recent health check results for this regional BackendService.
        # @param [String] project
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] backend_service
        #   Name of the BackendService resource for which to get health.
        # @param [Google::Apis::ComputeV1::ResourceGroupReference] resource_group_reference_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendServiceGroupHealth] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendServiceGroupHealth]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_backend_service_health(project, region, backend_service, resource_group_reference_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/backendServices/{backendService}/getHealth', options)
          command.request_representation = Google::Apis::ComputeV1::ResourceGroupReference::Representation
          command.request_object = resource_group_reference_object
          command.response_representation = Google::Apis::ComputeV1::BackendServiceGroupHealth::Representation
          command.response_class = Google::Apis::ComputeV1::BackendServiceGroupHealth
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a regional BackendService resource in the specified project using the
        # data included in the request. There are several restrictions and guidelines to
        # keep in mind when creating a regional backend service. Read  Restrictions and
        # Guidelines for more information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [Google::Apis::ComputeV1::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_region_backend_service(project, region, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/backendServices', options)
          command.request_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of regional BackendService resources available to the
        # specified project in the given region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::BackendServiceList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::BackendServiceList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_backend_services(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/backendServices', options)
          command.response_representation = Google::Apis::ComputeV1::BackendServiceList::Representation
          command.response_class = Google::Apis::ComputeV1::BackendServiceList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified regional BackendService resource with the data included
        # in the request. There are several restrictions and guidelines to keep in mind
        # when updating a backend service. Read  Restrictions and Guidelines for more
        # information. This method supports PATCH semantics and uses the JSON merge
        # patch format and processing rules.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to patch.
        # @param [Google::Apis::ComputeV1::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def patch_region_backend_service(project, region, backend_service, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:patch, '{project}/regions/{region}/backendServices/{backendService}', options)
          command.request_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Updates the specified regional BackendService resource with the data included
        # in the request. There are several restrictions and guidelines to keep in mind
        # when updating a backend service. Read  Restrictions and Guidelines for more
        # information.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region scoping this request.
        # @param [String] backend_service
        #   Name of the BackendService resource to update.
        # @param [Google::Apis::ComputeV1::BackendService] backend_service_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def update_region_backend_service(project, region, backend_service, backend_service_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:put, '{project}/regions/{region}/backendServices/{backendService}', options)
          command.request_representation = Google::Apis::ComputeV1::BackendService::Representation
          command.request_object = backend_service_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['backendService'] = backend_service unless backend_service.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves an aggregated list of commitments.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::CommitmentAggregatedList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::CommitmentAggregatedList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def aggregated_region_commitment_list(project, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/aggregated/commitments', options)
          command.response_representation = Google::Apis::ComputeV1::CommitmentAggregatedList::Representation
          command.response_class = Google::Apis::ComputeV1::CommitmentAggregatedList
          command.params['project'] = project unless project.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified commitment resource. Gets a list of available
        # commitments by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] commitment
        #   Name of the commitment to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Commitment] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Commitment]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_commitment(project, region, commitment, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/commitments/{commitment}', options)
          command.response_representation = Google::Apis::ComputeV1::Commitment::Representation
          command.response_class = Google::Apis::ComputeV1::Commitment
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['commitment'] = commitment unless commitment.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a commitment in the specified project using the data included in the
        # request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [Google::Apis::ComputeV1::Commitment] commitment_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_region_commitment(project, region, commitment_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/commitments', options)
          command.request_representation = Google::Apis::ComputeV1::Commitment::Representation
          command.request_object = commitment_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of commitments contained within the specified region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::CommitmentList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::CommitmentList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_commitments(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/commitments', options)
          command.response_representation = Google::Apis::ComputeV1::CommitmentList::Representation
          command.response_class = Google::Apis::ComputeV1::CommitmentList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns the specified regional disk type. Gets a list of available disk types
        # by making a list() request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] disk_type
        #   Name of the disk type to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskType] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskType]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_disk_type(project, region, disk_type, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/diskTypes/{diskType}', options)
          command.response_representation = Google::Apis::ComputeV1::DiskType::Representation
          command.response_class = Google::Apis::ComputeV1::DiskType
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['diskType'] = disk_type unless disk_type.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves a list of regional disk types available to the specified project.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::RegionDiskTypeList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::RegionDiskTypeList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_disk_types(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/diskTypes', options)
          command.response_representation = Google::Apis::ComputeV1::RegionDiskTypeList::Representation
          command.response_class = Google::Apis::ComputeV1::RegionDiskTypeList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Adds existing resource policies to a regional disk. You can only add one
        # policy which will be applied to this disk for scheduling snapshot creation.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] disk
        #   The disk name for this request.
        # @param [Google::Apis::ComputeV1::RegionDisksAddResourcePoliciesRequest] region_disks_add_resource_policies_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def add_region_disk_resource_policies(project, region, disk, region_disks_add_resource_policies_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/disks/{disk}/addResourcePolicies', options)
          command.request_representation = Google::Apis::ComputeV1::RegionDisksAddResourcePoliciesRequest::Representation
          command.request_object = region_disks_add_resource_policies_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a snapshot of this regional disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] disk
        #   Name of the regional persistent disk to snapshot.
        # @param [Google::Apis::ComputeV1::Snapshot] snapshot_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def create_region_disk_snapshot(project, region, disk, snapshot_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/disks/{disk}/createSnapshot', options)
          command.request_representation = Google::Apis::ComputeV1::Snapshot::Representation
          command.request_object = snapshot_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Deletes the specified regional persistent disk. Deleting a regional disk
        # removes all the replicas of its data permanently and is irreversible. However,
        # deleting a disk does not delete any snapshots previously made from the disk.
        # You must separately delete snapshots.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] disk
        #   Name of the regional persistent disk to delete.
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def delete_region_disk(project, region, disk, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:delete, '{project}/regions/{region}/disks/{disk}', options)
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns a specified regional persistent disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] disk
        #   Name of the regional persistent disk to return.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Disk] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Disk]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def get_region_disk(project, region, disk, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/disks/{disk}', options)
          command.response_representation = Google::Apis::ComputeV1::Disk::Representation
          command.response_class = Google::Apis::ComputeV1::Disk
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Creates a persistent regional disk in the specified project using the data
        # included in the request.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [Google::Apis::ComputeV1::Disk] disk_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] source_image
        #   Optional. Source image to restore onto a disk.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def insert_region_disk(project, region, disk_object = nil, request_id: nil, source_image: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/disks', options)
          command.request_representation = Google::Apis::ComputeV1::Disk::Representation
          command.request_object = disk_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['sourceImage'] = source_image unless source_image.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Retrieves the list of persistent disks contained within the specified region.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] filter
        #   A filter expression that filters resources listed in the response. The
        #   expression must specify the field name, a comparison operator, and the value
        #   that you want to use for filtering. The value must be a string, a number, or a
        #   boolean. The comparison operator must be either =, !=, >, or <.
        #   For example, if you are filtering Compute Engine instances, you can exclude
        #   instances named example-instance by specifying name != example-instance.
        #   You can also filter nested fields. For example, you could specify scheduling.
        #   automaticRestart = false to include instances only if they are not scheduled
        #   for automatic restarts. You can use filtering on nested fields to filter based
        #   on resource labels.
        #   To filter on multiple expressions, provide each separate expression within
        #   parentheses. For example, (scheduling.automaticRestart = true) (cpuPlatform = "
        #   Intel Skylake"). By default, each expression is an AND expression. However,
        #   you can include AND and OR expressions explicitly. For example, (cpuPlatform =
        #   "Intel Skylake") OR (cpuPlatform = "Intel Broadwell") AND (scheduling.
        #   automaticRestart = true).
        # @param [Fixnum] max_results
        #   The maximum number of results per page that should be returned. If the number
        #   of available results is larger than maxResults, Compute Engine returns a
        #   nextPageToken that can be used to get the next page of results in subsequent
        #   list requests. Acceptable values are 0 to 500, inclusive. (Default: 500)
        # @param [String] order_by
        #   Sorts list results by a certain order. By default, results are returned in
        #   alphanumerical order based on the resource name.
        #   You can also sort results in descending order based on the creation timestamp
        #   using orderBy="creationTimestamp desc". This sorts results based on the
        #   creationTimestamp field in reverse chronological order (newest result first).
        #   Use this to sort resources like operations so that the newest operation is
        #   returned first.
        #   Currently, only sorting by name or creationTimestamp desc is supported.
        # @param [String] page_token
        #   Specifies a page token to use. Set pageToken to the nextPageToken returned by
        #   a previous list request to get the next page of results.
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::DiskList] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::DiskList]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def list_region_disks(project, region, filter: nil, max_results: nil, order_by: nil, page_token: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:get, '{project}/regions/{region}/disks', options)
          command.response_representation = Google::Apis::ComputeV1::DiskList::Representation
          command.response_class = Google::Apis::ComputeV1::DiskList
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.query['filter'] = filter unless filter.nil?
          command.query['maxResults'] = max_results unless max_results.nil?
          command.query['orderBy'] = order_by unless order_by.nil?
          command.query['pageToken'] = page_token unless page_token.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Removes resource policies from a regional disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] disk
        #   The disk name for this request.
        # @param [Google::Apis::ComputeV1::RegionDisksRemoveResourcePoliciesRequest] region_disks_remove_resource_policies_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def remove_region_disk_resource_policies(project, region, disk, region_disks_remove_resource_policies_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/disks/{disk}/removeResourcePolicies', options)
          command.request_representation = Google::Apis::ComputeV1::RegionDisksRemoveResourcePoliciesRequest::Representation
          command.request_object = region_disks_remove_resource_policies_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Resizes the specified regional persistent disk.
        # @param [String] project
        #   The project ID for this request.
        # @param [String] region
        #   Name of the region for this request.
        # @param [String] disk
        #   Name of the regional persistent disk.
        # @param [Google::Apis::ComputeV1::RegionDisksResizeRequest] region_disks_resize_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def resize_region_disk(project, region, disk, region_disks_resize_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/disks/{disk}/resize', options)
          command.request_representation = Google::Apis::ComputeV1::RegionDisksResizeRequest::Representation
          command.request_object = region_disks_resize_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['disk'] = disk unless disk.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Sets the labels on the target regional disk.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The region for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::RegionSetLabelsRequest] region_set_labels_request_object
        # @param [String] request_id
        #   An optional request ID to identify requests. Specify a unique request ID so
        #   that if you must retry your request, the server will know to ignore the
        #   request if it has already been completed.
        #   For example, consider a situation where you make an initial request and the
        #   request times out. If you make the request again with the same request ID, the
        #   server can check if original operation with the same request ID was received,
        #   and if so, will ignore the second request. This prevents clients from
        #   accidentally creating duplicate commitments.
        #   The request ID must be a valid UUID with the exception that zero UUID is not
        #   supported (00000000-0000-0000-0000-000000000000).
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::Operation] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::Operation]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def set_region_disk_labels(project, region, resource, region_set_labels_request_object = nil, request_id: nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/disks/{resource}/setLabels', options)
          command.request_representation = Google::Apis::ComputeV1::RegionSetLabelsRequest::Representation
          command.request_object = region_set_labels_request_object
          command.response_representation = Google::Apis::ComputeV1::Operation::Representation
          command.response_class = Google::Apis::ComputeV1::Operation
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil?
          command.query['requestId'] = request_id unless request_id.nil?
          command.query['fields'] = fields unless fields.nil?
          command.query['quotaUser'] = quota_user unless quota_user.nil?
          command.query['userIp'] = user_ip unless user_ip.nil?
          execute_or_queue_command(command, &block)
        end
        
        # Returns permissions that a caller has on the specified resource.
        # @param [String] project
        #   Project ID for this request.
        # @param [String] region
        #   The name of the region for this request.
        # @param [String] resource
        #   Name or id of the resource for this request.
        # @param [Google::Apis::ComputeV1::TestPermissionsRequest] test_permissions_request_object
        # @param [String] fields
        #   Selector specifying which fields to include in a partial response.
        # @param [String] quota_user
        #   An opaque string that represents a user for quota purposes. Must not exceed 40
        #   characters.
        # @param [String] user_ip
        #   Deprecated. Please use quotaUser instead.
        # @param [Google::Apis::RequestOptions] options
        #   Request-specific options
        #
        # @yield [result, err] Result & error if block supplied
        # @yieldparam result [Google::Apis::ComputeV1::TestPermissionsResponse] parsed result object
        # @yieldparam err [StandardError] error object if request failed
        #
        # @return [Google::Apis::ComputeV1::TestPermissionsResponse]
        #
        # @raise [Google::Apis::ServerError] An error occurred on the server and the request can be retried
        # @raise [Google::Apis::ClientError] The request is invalid and should not be retried without modification
        # @raise [Google::Apis::AuthorizationError] Authorization is required
        def test_region_disk_iam_permissions(project, region, resource, test_permissions_request_object = nil, fields: nil, quota_user: nil, user_ip: nil, options: nil, &block)
          command = make_simple_command(:post, '{project}/regions/{region}/disks/{resource}/testIamPermissions', options)
          command.request_representation = Google::Apis::ComputeV1::TestPermissionsRequest::Representation
          command.request_object = test_permissions_request_object
          command.response_representation = Google::Apis::ComputeV1::TestPermissionsResponse::Representation
          command.response_class = Google::Apis::ComputeV1::TestPermissionsResponse
          command.params['project'] = project unless project.nil?
          command.params['region'] = region unless region.nil?
          command.params['resource'] = resource unless resource.nil