import 'package:dio/dio.dart';
import 'package:meta/meta.dart';
import 'package:peasmartcare/constraints/api_path.dart';
import 'package:peasmartcare/models/result_data_model.dart';
import 'package:peasmartcare/models/survey_answer_model.dart';
import 'package:peasmartcare/models/survey_group_model.dart';
import 'package:peasmartcare/models/survey_model.dart';
import 'package:peasmartcare/models/survey_question_model.dart';
import 'package:peasmartcare/models/survey_subject_model.dart';

abstract class SurveyService {
  Future<ResultData> getList({String authenKey, String surveyId});
  Future<ResultData> getSubject({String authenKey, String surveyId});
  Future<ResultData> getGroupList({String authenKey});
  Future<ResultData> getQuestion({String authenKey, String surveyId});
  Future<ResultData> sendAnswer({List<SurveyAnswerModel> answerList});
  Future<ResultData> deleteAnswer({String authenKey, String srcqId});
}

class SurveyImpService implements SurveyService {
  @override
  Future<ResultData> getList({@required String authenKey, String surveyId}) async {
    List<SurveyModel> resultList = List<SurveyModel>();
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.surveyList;
    Map<String, String> body = {
      "SQ_Id": surveyId,
      "authenKey": authenKey,
    };
    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        response.data.map((item) {
          SurveyModel result = SurveyModel.fromJson(item);
          resultList.add(result);
        }).toList();

        resultData.result = resultList;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getSubject({@required String authenKey, String surveyId}) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.surveySubject;
    Map<String, String> body = {
      "SQ_Id": surveyId,
      "authenKey": authenKey,
    };
    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        resultData.result = SurveySubjectModel.fromJson(response.data);
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getGroupList({@required String authenKey}) async {
    List<SurveyGroupModel> resultList = List<SurveyGroupModel>();
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.surveyGroupList;
    Map<String, String> body = {
      "authenKey": authenKey,
    };
    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        response.data.map((item) {
          SurveyGroupModel result = SurveyGroupModel.fromJson(item);
          resultList.add(result);
        }).toList();

        resultData.result = resultList;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getQuestion({@required String authenKey, String surveyId}) async {
    List<SurveyQuestionModel> resultList = List<SurveyQuestionModel>();
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.surveyQuestion;
    Map<String, String> body = {
      "SQ_Id": surveyId,
      "authenKey": authenKey,
    };
    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        response.data["QuestionDetail"].map((item) {
          SurveyQuestionModel result = SurveyQuestionModel.fromJson(item);
          resultList.add(result);
        }).toList();

        resultData.result = resultList;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> sendAnswer({List<SurveyAnswerModel> answerList}) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.surveySendAnswer;

    List<Map<String, dynamic>> body = answerList.map((item) => item.toJson()).toList();

    try {
      Response response = await dio.post(Uri.encodeFull(url), data: body);

      if (response.statusCode == 200) {
        resultData.result = true;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> deleteAnswer({@required String authenKey, String srcqId}) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.surveyDeleteAnswer;

    Map<String, String> body = {
      "SRCQ_Id": srcqId,
      "AuthenKey": authenKey,
    };

    try {
      Response response = await dio.post(Uri.encodeFull(url), data: body);

      if (response.statusCode == 200) {
        resultData.result = true;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }
}
