import 'package:dio/dio.dart';
import 'package:meta/meta.dart';
import 'package:peasmartcare/constraints/api_path.dart';
import 'package:peasmartcare/models/customer_ca.dart';
import 'package:peasmartcare/models/customer_ca_check.dart';
import 'package:peasmartcare/models/customer_info_model.dart';
import 'package:peasmartcare/models/electricity_location_customer_model.dart';
import 'package:peasmartcare/models/electricity_location_lat_long_model.dart';
import 'package:peasmartcare/models/result_data_model.dart';

abstract class ElectricityLocationService {
  Future<ResultData> getList({String authenKey, String idCard});
  Future<ResultData> getCurrentLocation({String authenKey, String customerId});
  Future<ResultData> saveCurrentLocation({
    String authenKey,
    String customerId,
    double latitude,
    double longitude,
  });
  Future<ResultData> customerCaCheck(
      {String authenKey, String caNumber, String peaNumber, String userId});
  Future<ResultData> getCustomerInfo({String authenKey, String customerId});
  Future<ResultData> create({String authenKey, String customerId});
  Future<ResultData> createCustomerContact({
    String customerId,
    String firstname,
    String lastname,
    String idCard,
    String phone,
    String authenKey,
  });
  Future<ResultData> delete(
      {String authenKey, String customerId, String idCard});
}

class ElectricityLocationImpService implements ElectricityLocationService {
  @override
  Future<ResultData> getList(
      {@required String authenKey, String idCard}) async {
    List<ElectricityLocationCustomerModel> resultList =
        List<ElectricityLocationCustomerModel>();
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.electricityLocationList;

    Map<String, String> body = {
      'idCard': idCard,
      "authenKey": authenKey,
    };
    print("url = ${url}");
    print("ListBody = ${body}");

    try {
      Response response =
          await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        print("response.statusCode = ${response.statusCode}");
        print("response.statusMessage = ${response.statusMessage}");
        print("response.data = ${response.data}");

        response.data.map((item) {
          ElectricityLocationCustomerModel customer =
              ElectricityLocationCustomerModel.fromJson(item);
          resultList.add(customer);
        }).toList();

        resultData.result = resultList;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getCurrentLocation(
      {@required String authenKey, String customerId}) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.electricityLocationGetCurrentLocation;

    Map<String, String> body = {
      'customerId': customerId,
      "authenKey": authenKey,
    };

    try {
      Response response =
          await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response != null &&
          response.statusCode == 200 &&
          response.data != null) {
        resultData.result =
            ElectricityLocationLatLongModel.fromJson(response.data);
      } else {
        resultData.result = null;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> saveCurrentLocation({
    @required String authenKey,
    String customerId,
    double latitude,
    double longitude,
  }) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.electricityLocationSaveCurrentLocation;

    Map<String, dynamic> body = {
      'latitude': latitude,
      'longtitude': longitude,
      'customerId': customerId,
      "authenKey": authenKey,
    };
    print("body = ${url}");
    print("body = ${body}");
    try {
      Response response =
          await dio.get(Uri.encodeFull(url), queryParameters: body);
      print("response = ${response.statusMessage}");
      print("response = ${response.statusCode}");
      print("response = ${response.data}");

      if (response.statusCode == 200) {
        resultData.result = true;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      print("catch = ${e}");

      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> customerCaCheck(
      {@required String authenKey,
      String caNumber,
      String peaNumber,
      String userId}) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.electricityLocationCheckCa;

    Map<String, String> body = {
      'CA': caNumber,
      'peaNo': peaNumber,
      "UserId": userId,
      "AuthenKey": authenKey,
    };
    print("url = ${url}");
    print("body = ${body}");

    try {
      Response response = await dio.post(Uri.encodeFull(url), data: body);
      print("statusMessage = ${response.statusMessage}");
      print("statusCodex = ${response.statusCode}");
      print("statusCode = ${response.extra}");
      print("statusCode = ${response.data}");
      // print("statusCode = ${response.realUri}");

      if (response.statusCode == 200) {
        resultData.result = CustomerCACheckModel.fromJson(response.data);
        // print("statusCode = ${resultData.result}");
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getCustomerInfo(
      {@required String authenKey, String customerId}) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.electricityLocationCustomerInfo;

    Map<String, String> body = {
      'customerId': customerId,
      "authenKey": authenKey,
    };

    try {
      Response response =
          await dio.get(Uri.encodeFull(url), queryParameters: body);
      print("bbb ${response.data}");
      print("statusCode ${response.statusCode}");

      if (response.statusCode == 200) {
        if (response.data["statusflag"] == null ||
            response.data["statusflag"] == true) {
          resultData.result = CustomerInfoModel.fromJson(response.data);
        } else {
          resultData.result = false;
        }
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> create(
      {@required String authenKey, String customerId}) async {
    CustomerCAModel customerCa;
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.electricityLocationCreate;

    Map<String, String> body = {
      'customerId': customerId,
      "authenKey": authenKey,
    };

    try {
      Response response = await dio.post(Uri.encodeFull(url), data: body);

      if (response.statusCode == 200) {
        response.data.map((item) {
          customerCa = CustomerCAModel.fromJson(item);
        }).toList();

        resultData.result = customerCa;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> createCustomerContact({
    String customerId,
    String firstname,
    String lastname,
    String idCard,
    String phone,
    @required String authenKey,
  }) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.electricityLocationCreateCustomerContact;

    Map<String, String> body = {
      'customerId': customerId,
      "customerContactName": firstname,
      "customerContactSurName": lastname,
      "customerContactCitizenNumber": idCard,
      "customerContactTel": phone,
      "authenKey": authenKey,
    };

    try {
      Response response = await dio.post(Uri.encodeFull(url), data: body);

      if (response.statusCode == 200) {
        if (response.data['statusCode'] == 200) {
          resultData.result = response.data['statusflag'];
          // resultData.statusMessage = response.data['Message'];
        } else {
          resultData.result = false;
          resultData.statusMessage = response.data['Message'];
        }
      } else {
        resultData.result = false;
        resultData.statusMessage = response.statusMessage;
      }

      resultData.statusCode = response.statusCode;
    } on DioError catch (e) {
      resultData.result = false;
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> delete(
      {@required String authenKey, String customerId, String idCard}) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.electricityLocationDelete;

    Map<String, String> body = {
      'customerId': customerId,
      'customerContactCitizenNumber': idCard,
      "AuthenKey": authenKey,
    };

    try {
      Response response = await dio.post(Uri.encodeFull(url), data: body);

      if (response.statusCode == 200) {
        resultData.result = true;
      } else {
        resultData.result = false;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }
}
