import 'package:dio/dio.dart';
import 'package:meta/meta.dart';
import 'package:peasmartcare/constraints/api_path.dart';
import 'package:peasmartcare/models/coordinator_model.dart';
import 'package:peasmartcare/models/coordinator_position_model.dart';
import 'package:peasmartcare/models/result_data_model.dart';

abstract class CoordinatorService {
  Future<ResultData> getList({String customerId, String authenKey});
  Future<ResultData> getDetail({String idCard, String authenKey});
  Future<ResultData> delete({String id, String authenKey});
  Future<ResultData> create({
    String customerId,
    String fullname,
    String position,
    String phone,
    String idCard,
    String otherContact,
    String authenKey,
  });
  Future<ResultData> update({
    String id,
    String customerId,
    String fullname,
    String position,
    String phone,
    String idCard,
    String otherContact,
    String authenKey,
  });
  Future<ResultData> getPositionList();
}

class CoordinatorImpService implements CoordinatorService {
  @override
  Future<ResultData> getList({String customerId, @required String authenKey}) async {
    List<CoordinatorModel> resultList = List<CoordinatorModel>();

    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.coordinatorList;

    Map<String, String> body = {
      "customerId": customerId,
      "authenKey": authenKey,
    };

    Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

    if (response.statusCode == 200) {
      response.data.map((item) {
        CoordinatorModel data = CoordinatorModel.fromJson(item);
        resultList.add(data);
      }).toList();

      resultData.result = resultList;
    }

    resultData.statusCode = response.statusCode;
    resultData.statusMessage = response.statusMessage;

    return resultData;
  }

  @override
  Future<ResultData> getDetail({String idCard, @required String authenKey}) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.coordinatorDetail;

    Map<String, String> body = {
      "citizenNumber": idCard,
      "AuthenKey": authenKey,
    };

    Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

    if (response.statusCode == 200) {
      resultData.result = CoordinatorModel.fromJson(response.data);
    }

    resultData.statusCode = response.statusCode;
    resultData.statusMessage = response.statusMessage;

    return resultData;
  }

  @override
  Future<ResultData> create({
    String customerId,
    String fullname,
    String position,
    String phone,
    String idCard,
    String otherContact,
    @required String authenKey,
  }) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.coordinatorCreate;

    Map<String, dynamic> body = {
      'CustomerId': customerId,
      "ContactName": fullname,
      'Position': position,
      'ContactTel': phone,
      'ContactEtc': otherContact,
      "CitizenNumber": idCard,
      "IsUsePeaSmartCare": true,
      'authenKey': authenKey,
    };

    Response response = await dio.post(Uri.encodeFull(url), data: body);

    if (response.statusCode == 200) {
      if (response.data['statusCode'] == 200) {
        resultData.result = true;
        // resultData.statusMessage = response.data['Message'];
      } else {
        resultData.result = false;
        resultData.statusMessage = response.data['Message'];
      }
    } else {
      resultData.result = false;
      resultData.statusMessage = response.statusMessage;
    }

    resultData.statusCode = response.statusCode;

    return resultData;
  }

  @override
  Future<ResultData> update({
    String id,
    String customerId,
    String fullname,
    String position,
    String phone,
    String idCard,
    String otherContact,
    @required String authenKey,
  }) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.coordinatorUpdate;

    Map<String, dynamic> body = {
      "CustomerContactInfoId": id,
      'CustomerId': customerId,
      "ContactName": fullname,
      'Position': position,
      'ContactTel': phone,
      'ContactEtc': otherContact,
      "CitizenNumber": idCard,
      "IsUsePeaSmartCare": true,
      'authenKey': authenKey,
    };

    try {
      Response response = await dio.post(Uri.encodeFull(url), data: body);

      if (response.statusCode == 200) {
        if (response.data['statusCode'] == 200) {
          resultData.result = true;
          // resultData.statusMessage = response.data['Message'];
        } else {
          resultData.result = false;
          resultData.statusMessage = response.data['Message'];
        }
        resultData.result = true;
      } else {
        resultData.result = false;
        resultData.statusMessage = response.statusMessage;
      }

      resultData.statusCode = response.statusCode;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> delete({String id, @required String authenKey}) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.coordinatorDelete;

    Map<String, String> body = {
      'customerContactInfoId': id,
      "authenKey": authenKey,
    };

    try {
      Response response = await dio.post(Uri.encodeFull(url), data: body);

      if (response.statusCode == 200) {
        resultData.result = true;
      } else {
        resultData.result = false;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
    }

    return resultData;
  }

  @override
  Future<ResultData> getPositionList() async {
    List<CoordinatorPositionModel> resultList = List<CoordinatorPositionModel>();

    ResultData resultData = ResultData();
    Response response;

    CoordinatorPositionModel position = CoordinatorPositionModel();
    position.id = "1";
    position.name = "ผู้จัดการ";
    resultList.add(position);

    position = CoordinatorPositionModel();
    position.id = "2";
    position.name = "โปรแกรมเมอร์";
    resultList.add(position);

    position = CoordinatorPositionModel();
    position.id = "3";
    position.name = "ผู้ช่วยผู้จัดการ";
    resultList.add(position);

    resultData.result = resultList;
    resultData.statusCode = 200;
    resultData.statusMessage = "OK";

    return resultData;
  }
}
