import 'package:dio/dio.dart';
import 'package:meta/meta.dart';
import 'package:peasmartcare/constraints/api_path.dart';
import 'package:peasmartcare/models/feedback_branch_model.dart';
import 'package:peasmartcare/models/feedback_detail_model.dart';
import 'package:peasmartcare/models/feedback_model.dart';
import 'package:peasmartcare/models/feedback_province_model.dart';
import 'package:peasmartcare/models/feedback_subject_suggestion_type_model.dart';
import 'package:peasmartcare/models/feedback_suggestion_type_model.dart';
import 'package:peasmartcare/models/feedback_type_model.dart';
import 'package:peasmartcare/models/file_model.dart';
import 'package:peasmartcare/models/result_data_model.dart';

abstract class FeedbackService {
  Future<ResultData> getFeedbackTypeList({String authenKey});
  Future<ResultData> getFeedbackProvinceList({String authenKey});
  Future<ResultData> getFeedbackBranchList({String authenKey, String provinceId});
  Future<ResultData> getFeedbackSuggestionTypeList({String authenKey, String feedbackTypeId});
  Future<ResultData> getFeedbackSubjectSuggestionTypeList({String authenKey, String suggestionTypeId});
  Future<ResultData> getList({String authenKey, String caNumber});
  Future<ResultData> getDetail({String authenKey, String feedbackId});
  Future<ResultData> save({
    String feedbackTypeId,
    String name,
    String provinceId,
    String branchId,
    String branchName,
    String contactLocation,
    String crimeScene,
    String suggestionTypeId,
    String subjectSuggestionTypeId,
    String detail,
    FileModel fileData,
    String caNumber,
    String authenKey,
  });
}

class FeedbackImpService implements FeedbackService {
  @override
  Future<ResultData> getFeedbackTypeList({@required String authenKey}) async {
    List<FeedbackTypeModel> resultList = List<FeedbackTypeModel>();
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.feedbackTypeList;

    Map<String, String> body = {
      "AuthenKey": authenKey,
    };

    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        response.data.map((item) {
          FeedbackTypeModel result = FeedbackTypeModel.fromJson(item);
          resultList.add(result);
        }).toList();

        resultData.result = resultList;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getFeedbackProvinceList({@required String authenKey}) async {
    List<FeedbackProvinceModel> resultList = List<FeedbackProvinceModel>();
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.feedbackProvinceList;

    Map<String, String> body = {
      "AuthenKey": authenKey,
    };

    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        response.data.map((item) {
          FeedbackProvinceModel result = FeedbackProvinceModel.fromJson(item);
          resultList.add(result);
        }).toList();

        resultData.result = resultList;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getFeedbackBranchList({@required String authenKey, String provinceId}) async {
    List<FeedbackBranchModel> resultList = List<FeedbackBranchModel>();
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.feedbackBranchList;

    Map<String, String> body = {
      "AuthenKey": authenKey,
      "provinceId": provinceId,
    };

    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        response.data.map((item) {
          FeedbackBranchModel result = FeedbackBranchModel.fromJson(item);
          resultList.add(result);
        }).toList();

        resultData.result = resultList;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getFeedbackSuggestionTypeList({@required String authenKey, String feedbackTypeId}) async {
    List<FeedbackSuggestionTypeModel> resultList = List<FeedbackSuggestionTypeModel>();
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.feedbackSuggestionTypeList;

    Map<String, String> body = {
      "AuthenKey": authenKey,
      "tagId": feedbackTypeId,
    };

    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        response.data.map((item) {
          FeedbackSuggestionTypeModel result = FeedbackSuggestionTypeModel.fromJson(item);
          resultList.add(result);
        }).toList();

        resultData.result = resultList;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getFeedbackSubjectSuggestionTypeList({@required String authenKey, String suggestionTypeId}) async {
    List<FeedbackSubjectSuggestionTypeModel> resultList = List<FeedbackSubjectSuggestionTypeModel>();
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.feedbackSubjectSuggestionTypeList;

    Map<String, String> body = {
      "AuthenKey": authenKey,
      "complaintTypeId": suggestionTypeId,
    };

    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        response.data.map((item) {
          FeedbackSubjectSuggestionTypeModel result = FeedbackSubjectSuggestionTypeModel.fromJson(item);
          resultList.add(result);
        }).toList();

        resultData.result = resultList;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getList({@required String authenKey, String caNumber}) async {
    List<FeedbackModel> resultList = List<FeedbackModel>();
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.feedbackList;

    Map<String, String> body = {
      "ca": caNumber,
      "authenKey": authenKey,
    };

    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        response.data.map((item) {
          FeedbackModel result = FeedbackModel.fromJson(item);
          resultList.add(result);
        }).toList();

        resultData.result = resultList;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> getDetail({@required String authenKey, String feedbackId}) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.feedbackDetail;

    Map<String, String> body = {
      "complaintId": feedbackId,
      "authenKey": authenKey,
    };

    try {
      Response response = await dio.get(Uri.encodeFull(url), queryParameters: body);

      if (response.statusCode == 200) {
        resultData.result = FeedbackDetailModel.fromJson(response.data);
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }

  @override
  Future<ResultData> save({
    String feedbackTypeId,
    String name,
    String provinceId,
    String idCard,
    String phone,
    String email,
    String branchId,
    String branchName,
    String contactLocation,
    String crimeScene,
    String suggestionTypeId,
    String subjectSuggestionTypeId,
    String detail,
    FileModel fileData,
    String caNumber,
    @required String authenKey,
  }) async {
    ResultData resultData = ResultData();
    Dio dio = Dio();

    String url = APIPath.feedbackSave;
    dynamic file = fileData != null
        ? [
            {
              "ExtensionData": fileData.extensionData,
              "FileContent": fileData.fileContentBase64,
              "FileName": fileData.fileName,
              "MimeType": fileData.mimeType,
            }
          ]
        : null;

    Map<String, dynamic> body = {
      "tagId": feedbackTypeId,
      "complainerName": name,
      "citizenNumber": idCard,
      "telephoneNumber": phone,
      "email": email,
      "provinceId": provinceId,
      "peaCode": branchId,
      "peaName": branchName,
      "accidentLocation": crimeScene,
      "caNumber": caNumber,
      "complaintTypeId": suggestionTypeId,
      "complaintSubtypeId": subjectSuggestionTypeId,
      "contactAddress": contactLocation,
      "detail": detail,
      "files": file,
      "authenKey": authenKey,
    };

    try {
      Response response = await dio.post(Uri.encodeFull(url), data: body);

      if (response.statusCode == 200) {
        resultData.result = true;
      } else {
        resultData.result = false;
      }

      resultData.statusCode = response.statusCode;
      resultData.statusMessage = response.statusMessage;
    } on DioError catch (e) {
      resultData.statusCode = e.response.statusCode;
      resultData.statusMessage = e.response.statusMessage;
    }

    return resultData;
  }
}
